//============================================================================
// CUSTOM RIGHT CLICK ACTIONS WHEN OVER A CANVAS CHILD
//============================================================================

// Disable the default right-click menu
document.getElementById('canvasContainer').addEventListener('contextmenu', function(event) {
  event.preventDefault();  // Prevent the default context menu
});

// Event listener to show custom context menu on right-click on a canvas
document.getElementById('canvasContainer').addEventListener('contextmenu', function(event) {
  const target = event.target;

  // Check if the right-clicked element is a canvas (canvas child)
  if (target.tagName === 'CANVAS') {
    // Show the custom context menu
    const contextMenu = document.getElementById('customContextMenu');
    contextMenu.style.display = 'block';
    contextMenu.style.left = `${event.pageX}px`;
    contextMenu.style.top = `${event.pageY}px`;

    // Store the clicked canvas element for later use (e.g., saving the image)
    contextMenu.currentCanvas = target;
  }
});

// Hide the context menu if clicked outside
document.addEventListener('click', function(event) {
  const contextMenu = document.getElementById('customContextMenu');
  contextMenu.style.display = 'none';
});

//====================================================================
//====================================================================

// Handle "Save Image" action
document.getElementById('saveImage').addEventListener('click', function () {
  const contextMenu = document.getElementById('customContextMenu');
  const canvas = contextMenu.currentCanvas; // Get the clicked canvas

  if (canvas) {
    // Retrieve the original filename from the canvas's dataset
	//console.log("FNAME:" + canvas.dataset.filename);
    const originalFilename = canvas.dataset.filename?.replace(/\.shp$/i, '') || 'Unknown'; // Handle undefined filename
	
    // Retrieve the original index from the canvas's dataset
	//console.log("FINDEX:" + canvas.dataset.index);
    const index = canvas.dataset.index;

    // Get the index of the canvas within the canvasContainer
    //const canvasContainer = document.getElementById('canvasContainer');
    //const canvases = Array.from(canvasContainer.getElementsByTagName('canvas'));
    //const index = canvases.indexOf(canvas) + 1; // 1-based index

    // Generate the dynamic file name using the original SHP filename and index
    const fileName = `${originalFilename}-${String(index).padStart(4, '0')}.png`;

    // Create a link element to trigger the download
    const link = document.createElement('a');
    link.href = canvas.toDataURL(); // Convert canvas to image data URL
    link.download = fileName; // Set the dynamic filename
    link.click();
  }

  // Hide the custom context menu after the action
  document.getElementById('customContextMenu').style.display = 'none';
});

//====================================================================
//====================================================================

// Handle "Copy Image to Clipboard" action
document.getElementById('copyImage').addEventListener('click', async function () {
  const contextMenu = document.getElementById('customContextMenu');
  const canvas = contextMenu.currentCanvas;

  if (canvas) {
    try {
      // Convert the canvas to a Blob
      canvas.toBlob(async (blob) => {
        const item = new ClipboardItem({ 'image/png': blob });
        await navigator.clipboard.write([item]); // Copy the blob to the clipboard
        //alert('Image copied to clipboard!');
      }, 'image/png');
    } catch (err) {
      console.error('Failed to copy image:', err);
      //alert('Failed to copy the image to clipboard.');
    }
  }
  document.getElementById('customContextMenu').style.display = 'none'; // Hide the menu
});

//====================================================================
//====================================================================

// Handle "Copy Image + Next Image with Divider to Clipboard" action
document.getElementById('copyDoubleImage').addEventListener('click', async function () {
  const contextMenu = document.getElementById('customContextMenu');
  const canvas = contextMenu.currentCanvas;

  if (canvas) {
    // Get the next canvas (if available)
    const canvasContainer = document.getElementById('canvasContainer');
    const canvases = Array.from(canvasContainer.getElementsByTagName('canvas'));
    const nextCanvas = canvases[canvases.indexOf(canvas) + 1]; // Get the next canvas

    if (!nextCanvas) {
      alert('No next image available to combine.');
      return;
    }

    // Create a new canvas to hold both images and the divider
    const combinedCanvas = document.createElement('canvas');
    const ctx = combinedCanvas.getContext('2d');

    // Set the dimensions of the combined canvas
    const combinedWidth = canvas.width + nextCanvas.width + 5; // 5 pixels for divider
    const combinedHeight = Math.max(canvas.height, nextCanvas.height); // Use the max height of both images
    combinedCanvas.width = combinedWidth;
    combinedCanvas.height = combinedHeight;

    // Draw the first image
    ctx.drawImage(canvas, 0, 0);

    // Draw the cyan divider (5 pixels wide)
    ctx.fillStyle = 'rgb(0, 255, 255)';
    ctx.fillRect(canvas.width, 0, 5, combinedHeight); // Draw a vertical divider

    // Draw the next image after the divider
    ctx.drawImage(nextCanvas, canvas.width + 5, 0);

    // Copy the combined image to the clipboard
    try {
      combinedCanvas.toBlob(async (blob) => {
        const item = new ClipboardItem({ 'image/png': blob });
        await navigator.clipboard.write([item]); // Copy the combined image blob to the clipboard
        //alert('Combined image copied to clipboard!');
      }, 'image/png');
    } catch (err) {
      console.error('Failed to copy image:', err);
      alert('Failed to copy the image to clipboard.');
    }
  }

  document.getElementById('customContextMenu').style.display = 'none'; // Hide the menu
});

//====================================================================
//====================================================================

// Handle "Double Save Image" action along with the next image and a divider
document.getElementById('saveDoubleImage').addEventListener('click', function () {
  const contextMenu = document.getElementById('customContextMenu');
  const canvas = contextMenu.currentCanvas; // Get the clicked canvas

  if (canvas) {
    // Retrieve the original filename from the canvas's dataset
    const originalFilename = canvas.dataset.filename?.replace(/\.shp$/i, '') || 'Unknown'; // Handle undefined filename

    // Retrieve the original index from the canvas's dataset
    const index = canvas.dataset.index;

    // Get the next canvas (if available)
    const canvasContainer = document.getElementById('canvasContainer');
    const canvases = Array.from(canvasContainer.getElementsByTagName('canvas'));
    const nextCanvas = canvases[canvases.indexOf(canvas) + 1]; // Get the next canvas

    if (!nextCanvas) {
      alert('No next image available to combine.');
      return;
    }

    // Create a new canvas to hold both images and the divider
    const combinedCanvas = document.createElement('canvas');
    const ctx = combinedCanvas.getContext('2d');

    // Set the dimensions of the combined canvas
    const combinedWidth = canvas.width + nextCanvas.width + 5; // 5 pixels for divider
    const combinedHeight = Math.max(canvas.height, nextCanvas.height); // Use the max height of both images
    combinedCanvas.width = combinedWidth;
    combinedCanvas.height = combinedHeight;

    // Draw the first image
    ctx.drawImage(canvas, 0, 0);

    // Draw the cyan divider (5 pixels wide)
    ctx.fillStyle = 'rgb(0, 255, 255)';
    ctx.fillRect(canvas.width, 0, 5, combinedHeight); // Draw a vertical divider

    // Draw the next image after the divider
    ctx.drawImage(nextCanvas, canvas.width + 5, 0);

    // Generate the dynamic file name using the original SHP filename and index
    const fileName = `${originalFilename}-${String(index).padStart(4, '0')}_combined.png`;

    // Create a link element to trigger the download
    const link = document.createElement('a');
    link.href = combinedCanvas.toDataURL(); // Convert combined canvas to image data URL
    link.download = fileName; // Set the dynamic filename
    link.click();
  }

  // Hide the custom context menu after the action
  document.getElementById('customContextMenu').style.display = 'none';
});

//====================================================================
//====================================================================

//---Handle Multiple events for downloading multiple icons

// Handle "Extract This Image in ZIP (Single Icon)" action
document.getElementById('downloadThis').addEventListener('click', () => {
  triggerZIPDownload(1); // Extract this single image only
});

// Handle "Extract This + Next Image in ZIP (Turreted Icon)" action
document.getElementById('downloadTwo').addEventListener('click', () => {
  triggerZIPDownload(2); // Extract this and the next image
});

// Handle "Extract This + Next N Images in ZIP (Multiple Icons)" action
document.getElementById('downloadNextN').addEventListener('click', () => {
  const n = parseInt(prompt("How many images to include (including this one)?", "1"), 10) || 1;
  triggerZIPDownload(n); // Extract 'n' images starting from the selected one
});

async function triggerZIPDownload(n) {
  const contextMenu = document.getElementById('customContextMenu');
  const canvas = contextMenu.currentCanvas; // Get the clicked canvas

  if (canvas) {
    const canvasContainer = document.getElementById('canvasContainer');
    const canvases = Array.from(canvasContainer.getElementsByTagName('canvas'));

    // Get the index of the clicked canvas
    const startIndex = canvases.indexOf(canvas);

    // Prepare ZIP file
    const zip = new JSZip();
    const csvContent = [];
    const palette = document.getElementById('paletteSelect').value;

    // Loop over the specified range of canvases
    for (let i = startIndex; i < startIndex + n && i < canvases.length; i++) {
      const currentCanvas = canvases[i];
      const dataURL = currentCanvas.toDataURL('image/png');

      // Retrieve filename and index from canvas dataset
      const originalFilename = currentCanvas.dataset.filename?.replace(/\.shp$/i, '') || 'Unknown';
      const imageIndex = currentCanvas.dataset.index || i + 1;

      // Generate the image filename
      const imageFilename = `${originalFilename}-${String(imageIndex).padStart(4, '0')}.png`;

      // Save the canvas image in the ZIP
      zip.file(imageFilename, dataURL.split(',')[1], { base64: true });

      // Prepare the metadata row
      const csvRow = [
        imageIndex, // SHP_Slot
        imageFilename,
        palette,
        imageHeights[i],
        imageWidths[i],
        imageVar1s[i],
        imageXStarts[i],
        imageYStarts[i],
        imageXEnds[i],
        imageYEnds[i],
      ].join(',');

      csvContent.push(csvRow);
    }

    // Create CSV content with a header row
    const csvHeader = 'SHP_Slot,imagefilename,palette,height,width,var1,xStart,yStart,xEnd,yEnd\n';
    const csvBlob = new Blob([csvHeader + csvContent.join('\n')], { type: 'text/csv' });
    zip.file('metadata.csv', csvBlob);

    // Generate ZIP and trigger download
    try {
      const content = await zip.generateAsync({ type: 'blob' });
      const link = document.createElement('a');
      link.href = URL.createObjectURL(content);
      link.download = `SHP_Images_${String(startIndex + 1).padStart(4, '0')}_to_${String(startIndex + n).padStart(4, '0')}.zip`;
      link.click();
    } catch (err) {
      console.error('Error generating ZIP:', err);
    }
  }

  // Hide the context menu
  document.getElementById('customContextMenu').style.display = 'none';
}

//=======================================================================
//Add replace image option in context menu
//=======================================================================

// Handle "Replace Image" action
document.getElementById('replaceImage').addEventListener('click', function() {
  const contextMenu = document.getElementById('customContextMenu');
  const canvas = contextMenu.currentCanvas;

  if (canvas) {
    // Create an input element to upload a new image
    const fileInput = document.createElement('input');
    fileInput.type = 'file';
    fileInput.accept = 'image/*'; // Accept any image type

    fileInput.onchange = function(event) {
      const file = event.target.files[0];
      if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
          const img = new Image();
          img.onload = function() {
            const ctx = canvas.getContext('2d');
            // Clear the existing canvas
            ctx.clearRect(0, 0, canvas.width, canvas.height);

            // Draw the new image on the canvas, resizing it to fit the canvas dimensions
            ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
          };
          img.src = e.target.result;  // Set the new image source
        };
        reader.readAsDataURL(file);  // Read the new image as data URL
      }
    };

    // Simulate a click on the file input to open the file dialog
    fileInput.click();
  }
  
  document.getElementById('customContextMenu').style.display = 'none';  // Hide the menu
}); 
