//===============================================================
// This event listener adds the button.
//===============================================================

document.getElementById('recompileSHPButton').addEventListener('click', function() {
  buildSHPFile();
});

//===============================================================
// This function quantitizes 16.7M images down to a 256-color indexed image
//===============================================================

function getClosestPaletteColorIndex(rgb, palette) {
  let minDistance = Number.MAX_VALUE;
  let closestIndex = 0;

  for (let i = 0; i < palette.length; i++) {
    const paletteColor = palette[i];
    const distance = Math.pow(rgb[0] - paletteColor[0], 2) +
                     Math.pow(rgb[1] - paletteColor[1], 2) +
                     Math.pow(rgb[2] - paletteColor[2], 2);
    if (distance < minDistance) {
      minDistance = distance;
      closestIndex = i;
    }
  }

  return closestIndex;
}

//===============================================================
// RLE compression function based on previous C++ logic
//===============================================================

function rleCompress(imageData, width, height, backgroundColorIndex) {
    const compressed = [];

    for (let y = 0; y < height; y++) {
        let x = 0; // Start at the beginning of the row

        while (x < width) {
            const currentPixel = imageData[y * width + x]; // Get the current pixel
            let count = 1; // Start counting the current pixel

            // Count the run of the same pixel
            while (x + count < width && imageData[y * width + (x + count)] === currentPixel && count < 0xFE) {
                count++;
            }

            // Handle the run
            if (currentPixel === backgroundColorIndex) {
                // If it's the background color
                if (count === width - x) {
                    compressed.push(1, 0); // Skip full line of background color
                } else {
                    compressed.push((count * 2) + 1); // Skip background pixels
                }
            } else {
                // If it's a run of actual colors
                compressed.push(count * 2); // Run length
                compressed.push(currentPixel); // Color index
            }

            // Move x to the next segment to process
            x += count; // Move past this run
        }

        // Handle the end of the line
        compressed.push(0); // Indicate end of line (this can be adjusted based on needs)
    }

    return new Uint8Array(compressed);
}

//===============================================================
// Build SHP file function
//===============================================================

function buildSHPFile() {
    const canvasContainer = document.getElementById('canvasContainer');
    const canvases = canvasContainer.getElementsByTagName('canvas');
    const header = [];
    const offsetTable = [];
    const compressedImages = [];
    let currentOffset = 8 + canvases.length * 8; // Offset for image table entries
    const backgroundColorIndex = 77; // Example background color index

    // Declare these variables outside the loop so they work damnit.
    let adjustedWidth, adjustedHeight, var1, xStart, yStart, xEnd, yEnd;

    Array.from(canvases).forEach((canvas, index) => {
        const ctx = canvas.getContext('2d');
        const imageData = ctx.getImageData(0, 0, canvas.width, canvas.height);
        const indexedData = new Uint8Array(imageData.width * imageData.height);

        // Convert the image to indexed colors based on the palette
        for (let i = 0; i < imageData.data.length; i += 4) {
            const rgb = [imageData.data[i], imageData.data[i + 1], imageData.data[i + 2]];
            indexedData[i / 4] = getClosestPaletteColorIndex(rgb, PALETTE);
        }

        // Get header values directly from the original canvas or image data
		//convert them to numbers...
        //const var1 = canvas.dataset.var1; // Set this to the appropriate value
        //const xStart = canvas.dataset.xStart; // Retrieve from data attributes or wherever stored
		//console.log("xStart:" + xStart);
		//const yStart = canvas.dataset.yStart;
        //const xEnd = canvas.dataset.xEnd;
        //const yEnd = canvas.dataset.yEnd;
		
		// Convert dataset values to numbers
		const var1 = parseInt(canvas.dataset.var1, 10); // Ensure it is a number
		const xStart = parseInt(canvas.dataset.xStart, 10); // Ensure it is a number
		const yStart = parseInt(canvas.dataset.yStart, 10); // Ensure it is a number
		const xEnd = parseInt(canvas.dataset.xEnd, 10); // Ensure it is a number
		const yEnd = parseInt(canvas.dataset.yEnd, 10); // Ensure it is a number
	
        // Find image bounds (subtracting 1 from height for yend)
        //const { xstart, ystart, xend, yend } = findImageBounds(indexedData, canvas.width, canvas.height, backgroundColorIndex);

        // Adjust width and height for compression
        //const adjustedWidth = xend - xstart + 1;
        //const adjustedHeight = yend - ystart + 1;
		
        const adjustedWidth = canvas.width +1;
        const adjustedHeight = canvas.height +1;
		
		// Log the values to confirm they are correct
		console.log(`Writing Image ${index + 1}:`);
		console.log(`Height: ${adjustedHeight - 1}, Width: ${adjustedWidth - 1}`);
		console.log(`var1: ${var1}, xStart: ${xStart}, yStart: ${yStart}, xEnd: ${xEnd}, yEnd: ${yEnd}`);
		
		//=======================================================
        // Compress the indexed image data using RLE
        const compressedData = rleCompress(indexedData, adjustedWidth, adjustedHeight, backgroundColorIndex);
		//=======================================================
		
        // Add to compressed images array
        compressedImages.push(new Uint8Array(compressedData));

        // Add entry to offset table
        offsetTable.push(currentOffset);
        currentOffset += compressedData.length + 24; // 24-byte header for each image
    });

    // Build the SHP header and combine everything
    const shpFileBuffer = new ArrayBuffer(currentOffset);
    const shpView = new DataView(shpFileBuffer);

    // Write the SHP header (version 1.10 and number of images)
	
    shpView.setUint8(0, '1'.charCodeAt(0)); // write header
    shpView.setUint8(1, '.'.charCodeAt(0)); // write header
    shpView.setUint8(2, '1'.charCodeAt(0)); // write header
    shpView.setUint8(3, '0'.charCodeAt(0)); // write header
    shpView.setUint32(4, canvases.length, true); //write image count 

    // Write the offset table
    for (let i = 0; i < offsetTable.length; i++) {
        shpView.setUint32(8 + i * 8, offsetTable[i], true);
        shpView.setUint32(8 + i * 8 + 4, 0, true); // Palette offset is ignored
    }

    // Write the compressed images
    let offset = 8 + canvases.length * 8;

    compressedImages.forEach((compressedData, index) => {
        // Write the 24-byte image header
        shpView.setUint16(offset, adjustedHeight - 1, true); // Height (minus 1)
        shpView.setUint16(offset + 2, adjustedWidth - 1, true);  // Width (minus 1)
        shpView.setInt32(offset + 4, var1, true); // var1
        shpView.setInt32(offset + 8, xStart, true); // xStart
        shpView.setInt32(offset + 12, yStart, true); // yStart
        shpView.setInt32(offset + 16, xEnd, true); // xEnd
        shpView.setInt32(offset + 20, yEnd, true); // yEnd
		
		//move offset to 24 to start writing compressed data.
        offset += 24;

        // Write the compressed image data
        for (let i = 0; i < compressedData.length; i++) {
            shpView.setUint8(offset++, compressedData[i]);
        }
    });

    // Create a blob and download the SHP file
    const blob = new Blob([shpFileBuffer], { type: 'application/octet-stream' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = 'recompiled.shp';
    link.click();
}
