/*
 * Dominions random map generator.
 *
 * Copyright (c) 2002 Keldon Jones
 *
 * Modified by Philippe Duchon (2004) for compatibility with Dominions 2
 *
 * Modified by Bryon "LintMan" Daly (2004) for assorted improvements and bug fixes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "dommap.h"


/*
* A 2-dimensional array containing elevation values for each
* pixel of the map.  The elevations will later be used to determine
* the terrain of each pixel.
*/
//static double **elevation;


/*
* Like above, only containing "richness" of each pixel.
*/
static double **richness;

/*
* Create the elevation and richness arrays.
*/
static void make_arrays(int size)
{
    int i;
    double * tEl;
    double * tRi;
    double * tSh;

    /* Make enough rows */
    //elevation = (double **)malloc(sizeof(double *) * size);
    //richness = (double **)malloc(sizeof(double *) * size);
    elevation = (double **)calloc(size, sizeof(double *));
    richness = (double **)calloc(size, sizeof(double *));
    if( opt.do_shade ) {
        //shadow = (double **)malloc(sizeof(double *) * size);
        shadow = (double **)calloc(size, sizeof(double *));
    }

    /* Make two arrays */
    //tEl = (double *) malloc( sizeof(double) * size * size);
    //tRi = (double *) malloc( sizeof(double) * size * size);
    //tSh = (double *) malloc( sizeof(double) * size * size);
    tEl = (double *) calloc(size * size, sizeof(double));
    tRi = (double *) calloc(size * size, sizeof(double));
    tSh = (double *) calloc(size * size, sizeof(double));

    /* Make each row */
    for (i = 0; i < size; i++)
    {
        /* Make this row */
        elevation[i] = tEl + i*size;
        richness[i] =  tRi + i*size;
        if( opt.do_shade ) {
            shadow[i] = tSh + i*size;
        }
    }
}

/*
* These functions use a "fractal" algorithm.  We begin by setting
* the elevation of each corner of the map to a random value.  Then
* the perform the following steps:
*
* 1) Set the elevation of each edge midpoint to the average of the
*    two nearby corners.
* 2) Add or subtract a random amount.
* 3) Set the elevation of the center to the average of the four
*    corners and likewise add or subtract a random amount.
*
* Then we divide the original square into four smaller squares and
* repeat the process until each square is only one pixel in size.
*/


/* Generates a uniform random number (0,1] */
double Urand()
{
    /*
    long x;
    double ret;

    x = rand();
    ret = (1.0 * x)/RAND_MAX;

    return 1.0 - ret;
    */
    // genrand_real1 returns a uniform real number in [0,1) range,
    // and we want (0,1]
    return (1.0 - genrand_real1());
}

/* Generate a Normal random variate */
static double Normrand()
{
    double x,y;

    // if y gets a 0, the log(y) below returns +INF,
    // which causes much grief.  Urand() should return
    // a value greater than 0, and less than or equal 
    // to 1, ie: (0,1]
    x = Urand();  // generates random (0,1]
    y = Urand();  // generates random (0,1]

    return (cos(2*M_PI*x) * sqrt(-2.0*log(y)));
}


/*
* Generate a random change amount.  This amount will be
* between -1.0 and 1.0.  The amount will be proportional to
* the ratio of "length" to "maxchg".  This will cause the
* changes at the end of the fractal process to be very small.
*
* Note: changed "max" to "maxchg", because apparently max is a 
 * template class and VC++ was getting confused here.
*/
static double random_change(int length, int maxchg, double ruggedness)
{
    /* 	long x; */
    double ret;

    /* 	/\* Generate a random number *\/ */
    /* 	x = rand(); */

    /* 	/\* Massage the number into a floating-point *\/ */
    /* 	ret = ((1.0 * x / RAND_MAX) - 0.5) * 2.0; */

    /* Normal random variate */
    ret = Normrand();
    
    /* Make numbers late in the process smaller */
    ret *= pow(1.0 * length / maxchg, 1.0 / ruggedness);

    /* Return result */
    return ret;
}

/*
* Generate an elevation/richness map.
*
* Size is the number of pixels along an edge, and must be of the form
* (2^n + 1).  Acceptable numbers are 129, 257, 513, etc.
*/

static void generate_field(double ** field, int size, double ruggedness)
{
    int sq_size;
    int i;
    int j;
    double value;

    size--;

    /* Fix values for the corners */
    field[0][0]       = random_change(1,1, ruggedness);
    field[0][size]    = random_change(1,1, ruggedness);
    field[size][0]    = random_change(1,1, ruggedness);
    field[size][size] = random_change(1,1, ruggedness);

    sq_size = size / 2;

    while (sq_size>=1)
    {
        /* Do the right/bottom borders first */
        for (i=0; i<size; i+= 2* sq_size)
        {
            /* Right border */
            value = (field[i][size] + field[i+2*sq_size][size])/2.0;
            field[i+sq_size][size] = value + random_change(sq_size, size+1, ruggedness);

            /* Bottom border */
            value = (field[size][i] + field[size][i+2*sq_size])/2.0;
            field[size][i+sq_size] = value + random_change(sq_size, size+1, ruggedness);
        }

        /* Now do the rest of the squares */
        for (i=0; i<size; i+=2*sq_size)
        {
            for (j=0; j<size; j+= 2*sq_size)
            {
                /* Top edge */
                value = (field[i][j] + field[i][j+2*sq_size])/2.0;
                field[i][j+sq_size] = value + random_change(sq_size, size+1, ruggedness);

                /* Left edge */
                value = (field[i][j] + field[i+2*sq_size][j])/2.0;
                field[i+sq_size][j] = value + random_change(sq_size,size+1, ruggedness);

                /* Square center */
                value = (field[i][j] + field[i][j+2*sq_size] +
                    field[i+2*sq_size][j] + 
                    field[i+2*sq_size][j+2*sq_size])/4.0;
                field[i+sq_size][j+sq_size] = value +
                    random_change(sq_size, size+1, ruggedness);
            }
        }
        sq_size /= 2;
    }
}


#ifdef OLD_GEN_FIELD
static void old_generate_field(double **field, int size)
{
    int sub_size = size / 2;
    int x, y;
    int x_i, y_i;
    double temp;

    /* Make the corners */
    field[0][0] = random_change(1, 1);
    field[0][size - 1] = random_change(1, 1);
    field[size - 1][0] = random_change(1, 1);
    field[size - 1][size - 1] = random_change(1, 1);

    /* Make smaller squares until they are size 1 */
    while (sub_size > 0)
    {
        /* Loop through each grid point */
        for (y = 0; y < size; y += sub_size)
        {
            /* Loop through each point on this row */
            for (x = 0; x < size; x += sub_size)
            {
                /* Compute step number */
                y_i = y / sub_size;
                x_i = x / sub_size;

                /* Skip points we've already set */
                if ((y_i % 2 == 0) && (x_i % 2 == 0))
                {
                    /* Already set */
                    continue;
                }

                /* Handle center-points */
                if ((y_i % 2 == 1) && (x_i % 2 == 1))
                {
                    /* Compute average of corners */
                    temp = (field[y - sub_size][x - sub_size] +
                        field[y - sub_size][x + sub_size] +
                        field[y + sub_size][x - sub_size] +
                        field[y + sub_size][x + sub_size]) / 4;

                    /* Set center */
                    field[y][x] = temp + random_change(sub_size, size);
                }

                /* Handle horizontal edges */
                else if (y_i % 2 == 0)
                {
                    /* Compute average of corners */
                    temp = (field[y][x - sub_size] +
                        field[y][x + sub_size]) / 2;

                    /* Set edge */
                    field[y][x] = temp + random_change(sub_size, size);
                }

                /* Handle vertical edges */
                else
                {
                    /* Compute average of corners */
                    temp = (field[y - sub_size][x] +
                        field[y + sub_size][x]) / 2;

                    /* Set edge */
                    field[y][x] = temp + random_change(sub_size, size);
                }
            }
        }

        /* Go through in half-size steps */
        sub_size /= 2;
    }
}
#endif // OLD_GEN_FIELD


/*
** Crop the unwanted edges off of a field.
*/
static void crop_field(double **field, int orig)
{
    int y, x, yoff, xoff;

    /* Compute amount to copy */
    yoff = (orig - opt.height) / 2;
    xoff = (orig - opt.width) / 2;

    /* Copy numbers to upper-left corner */
    for (y = 0; y < opt.height; y++)
    {
        for (x = 0; x < opt.width; x++)
        {
            /* Copy one number */
            field[y][x] = field[y + yoff][x + xoff];
        }
    }
}

/*
* Return percentage of field that will be under the the given level.
*/
static double get_percent_under(double **field, double level)
{
    int y, x;
    int count = 0;

    /* Loop through elevation map */
    for (y = 0; y < opt.height; y++)
    {
        for (x = 0; x < opt.width; x++)
        {
            /* Count pixels under level */
            if (field[y][x] < level) count++;
        }
    }

    /* Return percentage under level */
    return 100.0 * count / (opt.height * opt.width);
}

/*
* Pick a level such to force roughly the given amount of the map to be under
* the target level.
*/
static double get_divide_level(double **field, double target)
{
    double low = -2.0, high = 2.0, mid;
    double amt;

    /* Narrow down the level */
    while (high - low > 0.002)
    {
        /* Compute midpoint of levels */
        mid = (low + high) / 2;

        /* Find percentage under midpoint */
        amt = get_percent_under(field, mid);

        /* Check for too high */
        if (amt > target) high = mid;

        /* Too low */
        else low = mid;
    }

    /* Return good level */
    return (low + high) / 2;
}

/*
* Generate terrain (water, plains, forest, mountains) from the elevation
* and richness maps.
*
* Each pixel has an elevation and richness factor.  We use the following
* grid to determine the terrain:
*
*                  20%            60%            20%
*             +------------+---------------+-------------+
*       high  |  tundra    |    forest     |  mountain   |   20%
*   e         +------------+---------------+-------------+
*   l    mid  | wasteland  |    plains     |   forest    |   20%
*   e         +------------+---------------+-------------+
*   v    low  |   swamp    |    plains     |  farmland   |   25%
*             +------------+---------------+-------------+
*        sea  |   water    |    water      |    water    |   35%
*             +------------+---------------+-------------+
*                 poor           mid            rich
*
*                             richness
*
* We also add a small random factor to each, to provide a bit of
* dithering at the edges of each area.
*
* The percentages are modifiable via command-line options, but
* the contents of the grid aren't.  Should they be?
*/
static void generate_terrain(void)
{
    double elev_bound[3], rich_bound[2];
    double elev, rich;
    double shade;
    int y, x;

    /* First determine a good waterlevel */
    elev_bound[0] = get_divide_level(elevation, opt.water_level);

    /* Get a good forest level */
    elev_bound[1] = get_divide_level(elevation, opt.tree_level);

    /* Get a good mountain level */
    elev_bound[2] = get_divide_level(elevation, opt.mountain_level);

    /* Now determine a "poverty line" */
    rich_bound[0] = get_divide_level(richness, opt.poor_level);

    /* And a "rich line" */
    rich_bound[1] = get_divide_level(richness, opt.rich_level);

    /* Generate terrain for each pixel */
    for (y = 0; y < opt.height; y++)
    {
        for (x = 0; x < opt.width; x++)
        {
            /* Get this elevation */
            elev = elevation[y][x];

            /* Get this richness */
            rich = richness[y][x];

            /* Find minimum and maximum shadow for the map */
            if( opt.do_shade ) {
                shade = shadow[y][x];
                if( shade > max_shadow ) {
                    max_shadow = shade;
                }
                else if( shade < min_shadow ) {
                    min_shadow = shade;
                }
            } 

            /* Find minimum and maximum elevation for the map */
            if( elev > max_elevation ) {
                max_elevation = elev;
            }
            else if( elev < min_elevation ) {
                min_elevation = elev;
            }

            /* Under water */
            if (elev < elev_bound[0])
            {
                /* Make this water */
                map[y][x] = WATER;

                /* Go to next pixel */
                continue;
            }

            /* Add some randomness */
            elev += random_change(1, 1, opt.rugged) * opt.terrain_dither;
            rich += random_change(1, 1, opt.rugged) * opt.terrain_dither;

            /* Low elevation */
            if (elev < elev_bound[1])
            {
                /* Poor */
                if (rich < rich_bound[0]) map[y][x] = SWAMP;

                /* Normal */
                else if (rich < rich_bound[1]) map[y][x] = PLAINS;

                /* Rich */
                else map[y][x] = FARMLAND;
            }

            /* Medium elevation */
            else if (elev < elev_bound[2])
            {
                /* Poor */
                if (rich < rich_bound[0]) map[y][x] = WASTELAND;

                /* Normal */
                else if (rich < rich_bound[1]) map[y][x] = PLAINS;

                /* Rich */
                else map[y][x] = FOREST;
            }

            /* High elevation */
            else
            {
                /* Poor */
                if (rich < rich_bound[0]) map[y][x] = TUNDRA;

                /* Normal */
                else if (rich < rich_bound[1]) map[y][x] = FOREST;

                /* Rich */
                else map[y][x] = MOUNTAIN;
            }
        }
    }
}

/*
* Generate a map.
*/
void generate_map(void)
{
    int size;

    /* Round up the map size */
    for (size = 3; ; size = size * 2 - 1)
    {
        /* Check for big enough */
        if (size >= opt.height && size >= opt.width) break;
    }

    /* Make big enough arrays */
    make_arrays(size);

    /* Check verbosity */
    if (opt.verbose)
    {
        /* Message */
        printf("Generating elevation map...\n");
    }

    /* Generate a fractal elevation map */
    generate_field(elevation, size, opt.rugged);

    /* Check verbosity */
    if (opt.verbose)
    {
        /* Message */
        printf("Generating richness map...\n");
    }

    /* Generate a fractal richness map */
    generate_field(richness, size, opt.rugged);

    /* Generate a fractal shadow map */
    if (opt.do_shade) 
    {
        generate_field(shadow, size, opt.shade_ruggedness);
    }

    /* Crop both maps */
    crop_field(elevation, size);
    crop_field(richness, size);
    if (opt.do_shade) 
    {
        // BKD - don't we need to crop shadow map also?
        crop_field(shadow, size);  // lets try it
    }
    
    /* Check verbosity */
    if (opt.verbose)
    {
        /* Message */
        printf("Generating terrain from elevation and richness...\n");
    }

    /* Generate terrain from the maps */
    generate_terrain();
}
