/*
 * Dominions random map generator.
 *
 * Copyright (c) 2002 Keldon Jones
 *
 * Modified by Philippe Duchon (2004) for compatibility with Dominions 2
 *
 * Modified by Bryon "LintMan" Daly (2004) for assorted improvements and bug fixes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "dommap.h"


/*
* List of provinces.
*/
province *provs;
int num_provs;

/*
* Temporary array used for various things.
*
* This array is the size of the map.
*/
static int **temp_map;


/*
* Entry in the search queue or heap.
*/
typedef struct grid
{
    /* Location */
    int y;
    int x;

    /* Province */
    int prov;

    /* Cost */
    int cost;
} grid;

/*
* The search queue or heap.
*/
static grid *list;
static int queue_head, queue_tail, queue_size;
static int heap_num;

/*
* Store the deltas of the eight directions.
*/
int dy[8] = { -1, 0, 1, 0, -1, -1, 1, 1 };
int dx[8] = { 0, 1, 0, -1, -1, 1, -1, 1 };

/*
* Create the temporary map array.
*/
static void make_temp_map()
{
    int y;

    /* Make rows */
    temp_map = (int **)malloc(sizeof(int *) * opt.height);

    /* Make each row */
    for (y = 0; y < opt.height; y++)
    {
        /* Make this row */
        temp_map[y] = (int *)malloc(sizeof(int) * opt.width);
    }

    /* Create the search queue and heap */
    list = (grid *)malloc(sizeof(grid) * opt.height * opt.width);

    /* Store the queue size */
    queue_size = opt.height * opt.width;
}

/*
* Clear the temporary map by setting each element to zero.
*/
static void clear_temp_map()
{
    int y, x;

    /* Clear each row */
    for (y = 0; y < opt.height; y++)
    {
        /* Clear each element in the row */
        for (x = 0; x < opt.width; x++)
        {
            /* Clear this element */
            temp_map[y][x] = 0;
        }
    }
}

/*
* Initialize a province
*/
static void init_province(province *pp, int num)
{
    int i;

    /* Fake capital coordinates */
    pp->cap_x = -1;
    pp->cap_y = -1;

    /* Neither sea nor not-sea */
    pp->sea = -1;

    /* Shape defaults to circle */
    pp->type = PROV_CIRC;

    /* No pixels yet */
    for (i=0; i<TERRAIN_TYPE; i++)
        pp->pix[i] = 0;

    pp->total_pixels = 0;

    /* No neighbours yet */
    pp->neighbours = (int *)calloc((num+1), sizeof(int));
    pp->num_neighbours = 0;
    
    /* No deleted neighbours yet either */
    pp->del_nb = (int *)calloc((num+1), sizeof(int));
    pp->num_del_nb = 0;

    pp->connected = FALSE;
}

/*
* Make the province list.
*/
static void make_prov_list(int num)
{
    int i;

    /* Make the list itself */
    provs = (province *)malloc((1+num) * sizeof(province));

    /* Make the neighbour lists */
    for (i = 1; i < num; i++)
    {
        init_province( &provs[i], num);
    }
}

/*
* Clear the search queue.
*/
static void clear_queue(void)
{
    /* Reset head and tail */
    queue_head = queue_tail = 0;

    /* Reset heap variables */
    heap_num = 0;
}

/*
* Add a grid to the search queue.
*/
static void add_queue(int y, int x, int c, int p)
{
    /* Add the grid */
    list[queue_tail].y = y;
    list[queue_tail].x = x;
    list[queue_tail].cost = c;
    list[queue_tail].prov = p;

    /* One more element in the queue */
    queue_tail++;

    /* Check for wraparound */
    if (queue_tail == queue_size) queue_tail = 0;
}

/*
* Remove a grid from the queue.
*/
static grid remove_grid(void)
{
    grid g;

    /* Get the first grid */
    g = list[queue_head];

    /* Advance the queue head */
    queue_head++;

    /* Check for wraparound */
    if (queue_head == queue_size) queue_head = 0;

    /* Return the grid */
    return g;
}

/*
* Add a grid to the heap.
*
* We sort based on the "cost" field.
*/
static void add_heap(int y, int x, int c, int p)
{
    grid g;
    int n, parent;

    /* Add the grid to the bottom of the heap */
    list[heap_num].y = y;
    list[heap_num].x = x;
    list[heap_num].cost = c;
    list[heap_num].prov = p;

    /* Remember postition of newly added grid */
    n = heap_num;

    /* One more item in heap */
    heap_num++;

    /* Determine grid's parent */
    parent = (n - 1) / 2;

    /* Restore heap if necessary */
    while (list[n].cost < list[parent].cost)
    {
        /* Swap grid with parent */
        g = list[parent];
        list[parent] = list[n];
        list[n] = g;

        /* Remember new location and parent */
        n = parent;
        parent = (n - 1) / 2;
    }
}

/*
* Remove the lowest cost grid from the heap.
*/
static grid remove_heap(void)
{
    grid g, ret;
    int n, c1, c2;

    /* Remember the grid to be returned */
    ret = list[0];

    /* Bring the bottom-most grid to the top */
    list[0] = list[--heap_num];

    /* Start at new root */
    n = 0;

    /* Restore heap */
    while (1)
    {
        /* Check for bottom of heap */
        if (heap_num <= n * 2 + 1) break;

        /* Determine costs of children */
        c1 = list[n * 2 + 1].cost;
        c2 = list[n * 2 + 2].cost;

        /* Determine smaller side */
        if (c1 <= c2 || (heap_num == n * 2 + 2))
        {
            /* Swap with child if bigger */
            if (list[n].cost > c1)
            {
                /* Swap */
                g = list[n];
                list[n] = list[n * 2 + 1];
                list[n * 2 + 1] = g;

                /* Remember new location */
                n = n * 2 + 1;
            }
            else
            {
                /* Done */
                break;
            }
        }
        else
        {
            /* Swap with child if bigger */
            if (list[n].cost > c2)
            {
                /* Swap */
                g = list[n];
                list[n] = list[n * 2 + 2];
                list[n * 2 + 2] = g;

                /* Remember new location */
                n = n * 2 + 2;
            }
            else
            {
                /* Done */
                break;
            }
        }
    }

    /* Return remembered grid */
    return ret;
}


/*
* Return distance between two points.
*/
static int distance(int y1, int x1, int y2, int x2)
{
    int dy, dx;

    /* Compute deltas */
    dy = y1 - y2;
    dx = x1 - x2;

    /* Return distance */
    return (int) sqrt(dy * dy + dx * dx);
}


/*
* Mark either the land or sea pixels in the temp map.
*/
static void mark_land(int sea)
{
    int y, x;

    /* Mark each row */
    for (y = 0; y < opt.height; y++)
    {
        /* Mark each element in the row */
        for (x = 0; x < opt.width; x++)
        {
            /* Check for marking */
            if ((sea && map[y][x] == WATER) ||
                (!sea && map[y][x] != WATER))
            {
                /* Mark this grid */
                temp_map[y][x] = 1;

                /* Add this grid to the queue */
                add_queue(y, x, 0, 0);
            }
        }
    }
}

/*
* Mark the edges of the temp map.
*/
static void mark_edges()
{
    int y, x;

    /* Mark the left and right edges */
    for (y = 0; y < opt.height; y++)
    {
        /* Mark the left edge */
        temp_map[y][0] = 1;
        add_queue(y, 0, 0, 0);

        /* Mark the right edge */
        temp_map[y][opt.width - 1] = 1;
        add_queue(y, opt.width - 1, 0, 0);
    }

    /* Mark the top and bottom edges */
    for (x = 0; x < opt.width; x++)
    {
        /* Mark the top edge */
        temp_map[0][x] = 1;
        add_queue(0, x, 0, 0);

        /* Mark the bottom edge */
        temp_map[opt.height - 1][x] = 1;
        add_queue(opt.height - 1, x, 0, 0);
    }
}

/*
* Spread the marks out from the already marked areas out as far
* as the given distance.
*
* This marks areas near shores or the map edge so that capitals
* won't be placed near them.
*/
static void spread_mark(int distance)
{
    grid g;
    int y, x, dir;

    /* Spread while grids are in the queue */
    while (queue_head != queue_tail)
    {
        /* Get a grid */
        g = remove_grid();

        /* Done if we've reached the max distance */
        if (g.cost == distance) continue;

        /* Spread each direction */
        for (dir = 0; dir < 8; dir++)
        {
            /* Get new grid location */
            y = g.y + dy[dir];
            x = g.x + dx[dir];

            /* Check for out of bounds */
            if (y < 0 || y >= opt.height) continue;
            if (x < 0 || x >= opt.width) continue;

            /* Check for already marked */
            if (temp_map[y][x]) continue;

            /* Mark the new grid */
            temp_map[y][x] = 1;

            /* Add the new grid */
            add_queue(y, x, g.cost + 1, 0);
        }
    }
}

/*
* Mark an area near a newly placed capital.
*
* This prevents future capitals being placed too close.
*/
static void mark_circle(int y, int x, int r)
{
    int y1, y2, y3, x1, x2, x3;
    int d;
    int mct = 0;

    /* Determine borders of circle */
    y1 = y - r;
    y2 = y + r;
    x1 = x - r;
    x2 = x + r;

    /* Clip borders to size of map */
    if (y1 < 0) y1 = 0;
    if (x1 < 0) x1 = 0;
    if (y2 >= opt.height) y2 = opt.height - 1;
    if (x2 >= opt.width) x2 = opt.width - 1;

    /* Loop through area */
    for (y3 = y1; y3 <= y2; y3++)
    {
        for (x3 = x1; x3 <= x2; x3++)
        {
            /* Check distance */
            d = distance(y, x, y3, x3);

            /* Skip if distance too big */
            if (d > r) continue;

            /* Mark grid */
            temp_map[y3][x3] = 1;
            mct++;
        }
    }
    //printf("mark_circle: x %d   y %d   rad %d   count: %d\n", x, y, r, mct);
}

/*
* Place some capitals on unmarked areas on the map.
*
* After placing each capital, mark the area around it so that future
* capitals will not be placed too near.
*/
static int place_some_capitals(int num, int dis)
{
    int i, j, ind, left_off;
    int y, x;
    int placed = 0;
    int rind; // random index into array of unmarked pixels
    BOOL success, no_unmarked_left = FALSE;

    // if no unmarked pixels on our list, bail out.  (This shoudld never happen, though)
    if (queue_tail <= 0)
        return placed;

    left_off = 0;  // track where linear search left off (if necessary)

    /* Place requested number of capitals */
    for (i = 0; i < num; i++)
    {
        // if we think there is still some unmarked pixels, try to place a capital
        if (!no_unmarked_left)
        {
            success = FALSE;

            /* Try several times to place a capital */
            for (j = 0; j < 20000; j++)
            {
                /* 
                ** Get a random location
                ** The old way to randomly pick map pixels was hit-or miss
                ** and very frequently resulted in failing to find all 
                ** the requested water capitals.  This new way starts with
                ** an array of only the available, unmarked pixels, and selects
                ** randomly from that (much smaller) list.  It would be too 
                ** costly to rebuild the list to remove the newly marked pixels
                ** after each capital is selected, so we'll still need to 
                ** check for marked pixels, but there will be a far fewer 
                ** number of them, so we have much better odds of finding
                ** all the capitals we want.
                */
                //y = rand() % opt.height;
                //x = rand() % opt.width;
                rind = (int)(Urand() * (queue_tail-1));
                y = list[rind].y;
                x = list[rind].x;
                //printf("rind %d    x %d    y %d    map %d\n", rind, x, y, temp_map[y][x]);

                /* Check for marked */
                if (temp_map[y][x]) continue;

                /* Place a capital here */
                map[y][x] = CAPITAL;

                /* Mark a circle around the capital */
                mark_circle(y, x, dis);

                /* Done with this capital */
                placed++;
                success = TRUE;
                break;
            }
            // See if there's any remaining unmarked pixels
            if (!success)
            {
                if (opt.verbose)
                    printf("random search failed, trying linear search starting at %d\n", left_off);
                // random searching didn't turn up any unmarked provs,
                // now search linearly and use the first found.
                // if we get to the end of the list, set no_unmarked_left = TRUE
                // and we can skip looking for any further capitals
                for (ind = left_off; ind < queue_tail; ind++)
                {
                    y = list[ind].y;
                    x = list[ind].x;

                    /* Check for marked */
                    if (temp_map[y][x]) continue;

                    /* Place a capital here */
                    map[y][x] = CAPITAL;

                    /* Mark a circle around the capital */
                    mark_circle(y, x, dis);

                    /* Done with this capital */
                    placed++;
                    success = TRUE;
                    left_off = ind+1;  // remember where we left off to start there next time
                    if (opt.verbose)
                        printf("Success: found unmarked at %d\n", ind);
                    break;
                }
                // see if *that* search found something
                if (!success)
                {
                    // if the linear search failed, there are no unmarked pixels left
                    // so stop searching for capitals
                    no_unmarked_left = TRUE;
                    if (opt.verbose)
                        printf("No unmarked pixels remain available for capitals!\n");
                }
            }
        }
    }

    /* Return number placed */
    return placed;
}


/*
** Makes list of unmarked pixels (ie: marked by mark_land, mark edges, spread_mark)
** to be used for capital placement in place_some_capitals.
*/
static void get_unmasked_pixels(int type)
{
    int y, x;

    clear_queue();

    /* For each row */
    for (y = 0; y < opt.height; y++)
    {
        /* Find all pixels not marked, and add them to queue */
        for (x = 0; x < opt.width; x++)
        {
            /* Check for marked */
            if (temp_map[y][x]) 
                continue;
            else
            {
                /* 
                ** Add the unmarked grid to queue 
                ** We don't really need the grid/queue stuff, just an array of the unmarked points,
                ** but it's cheaper to reuse the queue rather than allocate a new array just 
                ** for this stuff.
                */
                add_queue(y, x, 0, 0);
            }
        }
    }

    /*
    printf("%d unmarked pixels found: validating...\n", queue_tail);
    for (int i = 0; i < queue_tail; i++)
    {
    y = list[i].y;
    x = list[i].x;

    if (temp_map[y][x])
    printf("i %d    x %d    y %d    map %d\n", i, x, y, temp_map[y][x]);
    }
    printf(" ...done\n");
    */
}


/*
* Place capitals on the map.
*
* First we place sea capitals, then land.
*
* We avoid placing capitals too near to shore, too near the edge of
* the map, or too near each other.
*/
static void place_capitals(void)
{
    int num_sea, num_land;

    /* Clear the temp map */
    clear_temp_map();

    /* Clear the search queue */
    clear_queue();

    // First, pick the sea capitals by masking off all land. 
    // too-near land, and edge pixels

    /* Mark all the land pixels */
    mark_land(LAND);

    /* Mark the edges of the map */
    mark_edges();

    /* Spread the marks out from the land using search queue */
    spread_mark(opt.shore_avoidance_sea);

    /* reuse search queue to hold list of all unmarked water pixels */
    get_unmasked_pixels(SEA);

    /* Place sea capitals */
    num_sea = place_some_capitals(opt.num_sea, opt.sea_spread);

    /* Check for failures */
    if (num_sea < opt.num_sea)
    {
        /* Print warning */
        fprintf(stderr, "Could only place %d out of %d sea provinces.\n",
            num_sea, opt.num_sea);
    }

    // Now repeat the process for land capitals, masking out water

    /* Clear the temp map */
    clear_temp_map();

    /* Clear the search queue */
    clear_queue();

    /* Mark all the sea pixels */
    mark_land(SEA);

    /* Mark the edges of the map */
    mark_edges();

    /* Spread the marks out from the sea using search queue */
    spread_mark(opt.shore_avoidance_land);

    /* reuse search queue to hold list of all unmarked land pixels */
    get_unmasked_pixels(LAND);

    /* Place land capitals */
    num_land = place_some_capitals(opt.num_land, opt.land_spread);

    /* Check for failures */
    if (num_land < opt.num_land)
    {
        /* Print warning */
        fprintf(stderr, "Could only place %d out of %d land provinces.\n",
            num_land, opt.num_land);
    }

    /* Hack -- Save number of provinces actually placed */
    opt.num_sea = num_sea;
    opt.num_land = num_land;

    /* Make the province list */
    make_prov_list(num_sea + num_land + 1);
}

/*
* Create the province list by scanning the map for capitals.
*
* We order the provinces in the same way that Dominions does.
*/
static void collect_capitals(void)
{
    int y, x, type;

    /* Scan the map */
    for (y = 0; y < opt.height; y++)
    {
        for (x = 0; x < opt.width; x++)
        {
            /* Check for capital */
            if (map[y][x] == CAPITAL)
            {
                /* Advance to next */
                num_provs++;

                /* Store this province */
                provs[num_provs].cap_y = y;
                provs[num_provs].cap_x = x;

                /* Hack -- determine if province is sea */
                if (y > 0 && map[y - 1][x] == WATER)
                {
                    /* Province is sea */
                    provs[num_provs].sea = 1;
                }
                else
                {
                    /* Province is land */
                    provs[num_provs].sea = 0;
                }


                /* Assume circular province */
                type = PROV_CIRC;

                /* Make some special shapes */
                //if (!provs[num_provs].sea &&
                //    (rand() % 100) < opt.amt_special)
                if ((genrand_real1() * 100.0) < opt.amt_special)  // use better random # gen
                {
                    /* Hack -- choose province type */
                    type = (int)(genrand_real1() * 3) + 1;
                }

                /* Store type */
                provs[num_provs].type = type;
            }
        }
    }
}

/*
* Return the distance from the capital pixel of a province.
*
* We "tweak" the distance if the province is supposed to have a
* special shape.
*/
static int distance_from_center(int prov, int y, int x)
{
    int py, px;
    int dy, dx;

    /* Get province center */
    py = provs[prov].cap_y;
    px = provs[prov].cap_x;

    /* Get deltas */
    dy = py - y;
    dx = px - x;

    /* Check for special shape */
    if (provs[prov].type == PROV_VERT)
    {
        /* Reduce delta in y-direction */
        dy /= 2;

        /* Increase delta in x-direction */
        dx *= 2;
    }
    else if (provs[prov].type == PROV_HORZ)
    {
        /* Reduce delta in x-direction */
        dx /= 2;

        /* Increase delta in y-direction */
        dy *= 2;
    }
    else if (provs[prov].type == PROV_BIG)
    {
        /* Reduce both deltas */
        dy /= 2;
        dx /= 2;
    }
    else if (provs[prov].type == PROV_SMALL)
    {
        /* Increase both deltas */
        dy *= 2;
        dx *= 2;
    }

    /* Return distance */
    return (int) sqrt(dy * dy + dx * dx);
}

/*
* Spread provinces out from the capitals.
*
* We mark what province each pixel belongs to on the temp map.
*
* We use the search heap instead of the queue, in order to always
* claim the next lowest-cost pixel.  Going from land to sea is
* very expensive, as is going far from the capital pixel.
*
* Currently, we prevent sea provinces spreading onto land, as
* Dominions often classifies provinces with any land whatsoever
* as a land province, even if it is mostly sea.
*/
static void spread_provinces(void)
{
    int i, y, x;
    grid g;
    int cost;

    /* Clear the temp map */
    clear_temp_map();

    /* Clear the queue */
    clear_queue();

    /* Add each capital to the heap */
    for (i = 1; i <= num_provs; i++)
    {
        /* Add the capital pixel to the heap */
        add_heap(provs[i].cap_y, provs[i].cap_x, 0, i);
    }

    /* Spread the provinces until we've claimed everything */
    while (heap_num)
    {
        /* Grab a grid */
        g = remove_heap();

        /* Check for already claimed */
        if (temp_map[g.y][g.x]) continue;

        /* Claim the grid */
        temp_map[g.y][g.x] = g.prov;

        /* Check each adjacent grid */
        for (i = 0; i < 4; i++)
        {
            /* Get grid coordinates */
            y = g.y + dy[i];
            x = g.x + dx[i];

            /* Skip out of bounds */
            if (y < 0 || y >= opt.height) continue;
            if (x < 0 || x >= opt.width) continue;

            /* Skip already claimed pixels */
            if (temp_map[y][x]) continue;

            /* Don't allow sea provinces to extend to land */
            // BKD - this leaves unprovinced land islets in the water, which play hell 
            // with the impassable border stuff.  The stuff elsewhere about needing to not have 
            // any land pixels in a sea province is no longer true 
            //if (map[y][x] != WATER && provs[g.prov].sea) 
            //    continue; 

            /* Compute cost based on distance */
            cost = distance_from_center(g.prov, y, x);

            /* Add cost if crossing land/sea boundary */
            if ((map[g.y][g.x] != CAPITAL) &&
                ((map[g.y][g.x] == WATER && map[y][x] != WATER) ||
                (map[g.y][g.x] != WATER && map[y][x] == WATER)))
            {
                /* High cost */
                if (map[g.y][g.x] != WATER)  // BKD - experiment with diff costs for land->sea and sea->land
                    cost += opt.sea_cost;
                else
                    cost += 15*opt.sea_cost;
            }

            /* Multiply cost for other terrain crossing */
            else if (cost > 5 && map[g.y][g.x] != map[y][x])
            {
                /* Increase cost */
                cost *= opt.terrain_cost;
            }

            /* Add the grid to the heap */
            add_heap(y, x, g.cost + cost, g.prov);
        }
    }
}

/*
* Smooth province borders.
*
* We sort of "average" the pixels by taking the most common neighbor
* for each pixel on the map.  This removes extraneous borders that
* creep into the edge of a province.
*
* Note that this algorithm sorta steps on its own data as it goes.
* Maybe we need a temp_map2 array, but the output looks OK without
* going to that.
*
* This function may be called multiple times, if the "smoothing"
* option is greater than 1.
*/
static void smooth_borders(void)
{
    int y, x, i, j;
    int ny, nx, p;
    int nearby[9], count[9], num;
    int b_i, b_c;

    /* Loop over the map */
    for (y = 1; y < opt.height - 1; y++)
    {
        for (x = 1; x < opt.width - 1; x++)
        {
            /* Clear counts */
            for (i = 0; i < 9; i++) count[i] = 0;

            /* Count self */
            nearby[0] = temp_map[y][x];
            count[0] = 1;

            /* Assume no neighbors yet */
            num = 1;

            /* Check neighbors */
            for (i = 0; i < 8; i++)
            {
                /* Get neighbor pixel */
                ny = y + dy[i];
                nx = x + dx[i];

                /* Collect nearby province */
                p = temp_map[ny][nx];

                /* Find bucket */
                for (j = 0; j < num; j++)
                {
                    /* Check for match */
                    if (p == nearby[j]) break;
                }

                /* Fill bucket */
                nearby[j] = p;
                count[j]++;

                /* Increment number of buckets if necessary */
                if (j == num) num++;
            }

            /* Assume none found */
            b_i = b_c = -1;

            /* Find biggest bucket */
            for (i = 0; i < num; i++)
            {
                /* Check for bigger count */
                if (count[i] > b_c)
                {
                    /* Remember bucket */
                    b_i = i;
                    b_c = count[i];
                }
            }

            /* Change to most common neighbor */
            temp_map[y][x] = nearby[b_i];
        }
    }
}

/*
* Force all sea province pixels to water.
*
* This only needs to be called if smooth_borders has been called
* at least once.
*
* I don't want to have this function, but Dominions thinks that a
* province with even 2% land is a land province.  So to be safe,
* we need the sea provinces to be completely water.
*/
static void force_water(void)
{
    int y, x, p;

    /* Loop over map */
    for (y = 0; y < opt.height; y++)
    {
        for (x = 0; x < opt.width; x++)
        {
            /* Get province number */
            p = temp_map[y][x];

            /* Check for land */
            if (!provs[p].sea) continue;

            /* Skip capitals */
            if (map[y][x] == CAPITAL) continue;

            /* Force to sea */
            map[y][x] = WATER;
        }
    }
}

/*
* Add a neighbour relationship to the province list.
*/
static void add_neighbour(int prov, int neigh)
{
    int i;

    /* Loop through current neighbours */
    for (i = 0; i < provs[prov].num_neighbours; i++)
    {
        /* Check for match */
        if (provs[prov].neighbours[i] == neigh)
        {
            /* No need to do anything */
            return;
        }
    }

    /* Add a new neighbour */
    provs[prov].neighbours[i] = neigh;
    provs[prov].num_neighbours++;
}

/*
* Collect province neighbouring information.
*
* Neighbor information is stored twice, ie. if province 1 is a neighbour
* of province 2, this is stored in both 1 and 2's province information.
*/
static void collect_neighbours(void)
{
    int y, x, d;
    int ny, nx;

    /* Scan the map */
    for (y = 1; y < opt.height - 1; y++)
    {
        for (x = 1; x < opt.width - 1; x++)
        {
            /* Skip pixels not in any province */
            if (!temp_map[y][x]) continue;

            /* Check neighbours of this pixel */
            for (d = 0; d < 4; d++)
            {
                /* Get neighbour coordinates */
                ny = y + dy[d];
                nx = x + dx[d];

                /* Skip neighbours not in any province */
                if (!temp_map[ny][nx]) continue;

                /* Check for different province */
                if (temp_map[y][x] != temp_map[ny][nx])
                {
                    /* Add neighbour association */
                    add_neighbour(temp_map[y][x],
                        temp_map[ny][nx]);
                }
            }
        }
    }
}

/* 
** deletes the neighbour link from province prov to neighbor_province
** as long as this won't strand either province
** Note this only deletes the connection in the one direction,
** and a second call to this function from the neighbour to this prov
** should also be done.
*/
static void delete_neighbour(int prov, int neighbor_province)
{
    int nb;
    BOOL found = FALSE;

#ifdef DEBUG_IMPASS
    /* dump orig neighbours */
    printf("%d old connections %d =>", provs[prov].num_neighbours, prov);
    for (nb = 0; nb < provs[prov].num_neighbours; nb++)
    {
        printf(" %d", provs[prov].neighbours[nb]);
    }
    printf("\n");
    printf("%d old deleted connections %d =>", provs[prov].num_del_nb, prov);
    for (nb = 0; nb < provs[prov].num_del_nb; nb++)
    {
        printf(" %d", provs[prov].del_nb[nb]);
    }
    printf("\n");
#endif
    
    // don't delete link if this will leave a province with 0 connections
    if (provs[prov].num_neighbours < 2)
        return;
    
    /* Loop through current neighbours */
    for (nb = 0; nb < provs[prov].num_neighbours; nb++)
    {
        /* Check for match, and only delete if this won't strand the neighbor province */
        if ((provs[prov].neighbours[nb] == neighbor_province) && (provs[prov].num_neighbours >= 2))
        {
            //if (opt.verbose) printf("deleting %d => %d connection\n", prov, neighbor_province);
            found = TRUE;
            provs[prov].del_nb[provs[prov].num_del_nb++] = neighbor_province;  // remember deleted link
            provs[prov].num_neighbours--;
        }

        /* if found match, shift rest of links down one spot */
        if ((found) && (nb < provs[prov].num_neighbours))
        {
            //printf("shifting neighbor %d to ind %d\n", provs[prov].neighbours[nb+1], nb);
            provs[prov].neighbours[nb] = provs[prov].neighbours[nb+1];
        }
    }
    if (!found)
    {
        fprintf(stderr, "***** Warning: requested deletion link not found: %d -> %d!\n",  prov, neighbor_province);
    }

#ifdef DEBUG_IMPASS
    /* dump new neighbours */
    printf("%d new connections %d =>", provs[prov].num_neighbours, prov);
    for (nb = 0; nb < provs[prov].num_neighbours; nb++)
    {
        printf(" %d", provs[prov].neighbours[nb]);
    }
    printf("\n");
    printf("%d new deleted connections %d =>", provs[prov].num_del_nb, prov);
    for (nb = 0; nb < provs[prov].num_del_nb; nb++)
    {
        printf(" %d", provs[prov].del_nb[nb]);
    }
    printf("\n");
#endif
}



/*
** Looks for deleted link from prov->neighbor_province, and restores it
** as a neighbor.
** Note this only undeletes the connection in the one direction,
** and a second call to this function from the neighbour to this prov
** should also be done.
*/
static void undelete_neighbour(int prov, int neighbor_province)
{
    int nb;
    BOOL found = FALSE;

#ifdef DEBUG_IMPASS
    /* dump orig neighbours */
    printf("%d old connections %d =>", provs[prov].num_neighbours, prov);
    for (nb = 0; nb < provs[prov].num_neighbours; nb++)
    {
        printf(" %d", provs[prov].neighbours[nb]);
    }
    printf("\n");
    printf("%d old deleted connections %d =>", provs[prov].num_del_nb, prov);
    for (nb = 0; nb < provs[prov].num_del_nb; nb++)
    {
        printf(" %d", provs[prov].del_nb[nb]);
    }
    printf("\n");
#endif
    
    /* Loop through current deleted neighbours */
    for (nb = 0; nb < provs[prov].num_del_nb; nb++)
    {
        /* Check for match, and only delete if this won't strand the neighbor province */
        if ((provs[prov].del_nb[nb] == neighbor_province))
        {
            //if (opt.verbose) printf("restoring %d => %d connection\n", prov, neighbor_province);
            found = TRUE;
            provs[prov].neighbours[provs[prov].num_neighbours++] = neighbor_province; // add connection back in
            provs[prov].num_del_nb--;  // one less deleted connection
        }

        /* if found match, shift rest of deleted links down one spot */
        if ((found) && (nb < provs[prov].num_del_nb))
        {
            //printf("shifting del neighbor %d to idx %d\n", provs[prov].del_nb[nb+1], nb);
            provs[prov].del_nb[nb] = provs[prov].del_nb[nb+1];
        }
    }
    if (!found)
    {
        fprintf(stderr, "***** Warning: requested undeletion link not found: %d -> %d!\n",  prov, neighbor_province);
    }

#ifdef DEBUG_IMPASS
    /* dump new neighbours */
    printf("%d new connections %d =>", provs[prov].num_neighbours, prov);
    for (nb = 0; nb < provs[prov].num_neighbours; nb++)
    {
        printf(" %d", provs[prov].neighbours[nb]);
    }
    printf("\n");
    printf("%d new deleted connections %d =>", provs[prov].num_del_nb, prov);
    for (nb = 0; nb < provs[prov].num_del_nb; nb++)
    {
        printf(" %d", provs[prov].del_nb[nb]);
    }
    printf("\n");
#endif
}



static void check_connectedness()
{
    int *prov_q = malloc((num_provs + 1) * sizeof(int));  // list of provs to follow links on
    int q_hd = 0, q_tl = 0;  // province queue head/tail
    int cur_prov, nb;
    int i, j;
    BOOL found_unconnected;
    BOOL progress_made;
    BOOL reconnected;

    // Start out following connections with province #1
    prov_q[q_tl++] = 1;
    provs[1].connected = TRUE;  // mark this prov as connected so we don't re-follow it
    //printf("Adding prov 1 to connection queue\n");

    do
    {
        // clear these flags here on every pass - they will be set as appropriate below
        found_unconnected = FALSE;  
        progress_made = FALSE; 

        // while we have entries on the queue
        while (q_hd < q_tl)
        {
            /* Dump connected province queue/list
            printf("p_q = ");
            for (i = 0; i < q_tl; i++)
            {
                if (i == q_hd)
                {
                    printf("*"); // put a star next to queue head
                }
                printf("%d ", prov_q[i]);
            }
            printf("\n");
            */
            
            cur_prov = prov_q[q_hd++];  // "pop" next prov in queue
            //printf("Following prov %d\n", cur_prov);

            // add all neighbors to queue unless they've already been followed
            for (i = 0; i < provs[cur_prov].num_neighbours; i++)
            {
                nb = provs[cur_prov].neighbours[i];
                // if neighbor isn't already "connected"
                if (!provs[nb].connected)
                {
                    // add neighbor to queue
                    prov_q[q_tl++] = nb;
                    provs[nb].connected = TRUE;  // mark this prov as connected so we don't re-follow it
                    progress_made = TRUE;  // progress: a new entry in connection queue
                    //printf("Adding prov %d to connection queue\n", nb);
                }
                else 
                {
                    //printf("prov %d already connected\n", nb);
                }
            }
        }

        // At this point, if the graph is fully connected, every province should have connected = TRUE.
        // We search through looking for any provs that aren't connected and if we find any, we can try to 
        // restore a deleted neighbor link to reconnect that province to the main graph.  
        for (cur_prov = 1; cur_prov <= num_provs; cur_prov++)
        {
            reconnected = FALSE;
            // if we find an unconnected province
            if (!provs[cur_prov].connected)
            {
                //printf("*****\nFound unconnected province %d\n*****\n", cur_prov);
                found_unconnected = TRUE;  

                // look through its deleted neighbors to see if any are already connected
                // if so, restore the connection, if not, we pass it by and hope that
                // we'll susequently reconnect one if *its* neighbors so that both will be 
                // connected.  To avoid looping forever, we will stop if we loop through all 
                // provinces without the "progress_made" flag getting set.
                for (i = 0; i < provs[cur_prov].num_del_nb; i++)
                {
                    // see if this deleted neighbor is connected by searching prov_q
                    for (j = 0; j <= q_tl; j++)
                    {
                        // if we find a match
                        if (provs[cur_prov].del_nb[i] == prov_q[j])
                        {
                            //printf("undeleting link %d <-> %d\n", provs[cur_prov].del_nb[i], cur_prov);
                            
                            // found a previously deleted connection from our unconnected prov  
                            // to the main graph, so restore the connection in both directions
                            undelete_neighbour(provs[cur_prov].del_nb[i], cur_prov);  // undelete neighbor's link to us
                            undelete_neighbour(cur_prov, provs[cur_prov].del_nb[i]);  // undelete our link to neighbor

                            // now add the newly connected prov to the tail of connection queue
                            prov_q[q_tl++] = cur_prov;
                            provs[cur_prov].connected = TRUE;  // mark this prov as connected so we don't re-follow it
                            //printf("Adding prov %d to connection queue\n", cur_prov);

                            progress_made = TRUE; // progress: a new entry in connection queue
                            reconnected = TRUE; // we made a reconnection, start over from top
                            break;
                        }
                    }
                    if (reconnected)  // if we made a reconnection, fall out of province chekcing to get back to top
                        break;
                }
                if (reconnected)  // if we made a reconnection, fall out of province chekcing to get back to top
                    break;
            }
        }
    } while (found_unconnected && progress_made);  // keep looping until all connected or we stop progressing

    if (found_unconnected)
    {
        // if we got here and are still unconnected, it means we gave up from lack of progress.  In theory, this 
        // shouldn't ever happen...
        fprintf(stderr, "**** Warning: unable to fully connect all provinces\n");
    }

    free(prov_q);  // free up allocated memory
}


static void create_impasses(void)
{
    int prov, nb;

    /* Go through every province (note provs start counting at 1) */
    for (prov = 1; prov <= num_provs; prov++)
    {
        /* Loop through current neighbours */
        for (nb = 0; nb < provs[prov].num_neighbours; nb++)
        {
            // if at least 2 links remain, roll dice to see if we should delete this link
            if ((provs[prov].num_neighbours >= 2) && (opt.impass > Urand()))
            {
                int neigh = provs[prov].neighbours[nb];  // get province # of this neighbor

                // if this connection is sea-to-sean and sea-sea impassable borders not allowed
                // then bail out
                if (!opt.sea_sea_impass && provs[prov].sea && provs[neigh].sea)
                    continue;

                /* make sure that neighbor won't get stranded if we delete this connection */
                if (provs[neigh].num_neighbours >= 2)
                {
                    delete_neighbour(neigh, prov); // delete neighbor's link to us
                    delete_neighbour(prov, neigh); // delete our link to neighbor
                }
            }
        }
    }

    if (opt.verbose)
        printf("Verifying map province connectivity...\n");
    check_connectedness();  // forces map to be fully connected or your money back :-)
}


/*
* Draw borders around each province.
*
* We place a border pixel on any space that borders a province number
* higher than the current province.  This prevents double-thick borders.
*/
static void draw_borders(void)
{
    int y, x, i, j, border_type;
    int ny, nx, p, q;
    BOOL found_neighbour = FALSE;

    /* Draw borders around the edges of the map */
    /* Removed, as Dom2 doesn't need them, plus would prevent
    wraparound maps */
    /* 	for (y = 0; y < opt.height; y++) */
    /* 	{ */
    /* 		/\* Left edge *\/ */
    /* 		map[y][0] = LAND_BORDER; */

    /* 		/\* Right edge *\/ */
    /* 		map[y][opt.width - 1] = LAND_BORDER; */
    /* 	} */

    /* 	for (x = 0; x < opt.width; x++) */
    /* 	{ */
    /* 		/\* Top edge *\/ */
    /* 		map[0][x] = LAND_BORDER; */

    /* 		/\* Bottom edge *\/ */
    /* 		map[opt.height - 1][x] = LAND_BORDER; */
    /* 	} */

    /* Scan the map */
    for (y = 0; y < opt.height; y++)
    {
        for (x = 0; x < opt.width; x++)
        {
            /* Scan adjacent pixels */
            for (i = 0; i < 4; i++)
            {
                /* Get coordinates */
                ny = y + dy[i];
                nx = x + dx[i];

                /* Skip out of bounds */
                if (ny < 0 || ny >= opt.height) continue;
                if (nx < 0 || nx >= opt.width) continue;

                /* Skip boring neighbours */
                if (temp_map[ny][nx] <= temp_map[y][x])
                {
                    continue;
                }

                /* Get province numbers */
                p = temp_map[y][x];
                q = temp_map[ny][nx];

                /* look for neighbour link from p->q, if it doesn't exist, */
                /* then make border impassable */
                found_neighbour = FALSE;

                for (j = 0; j < provs[p].num_neighbours; j++)
                {
                    if (provs[p].neighbours[j] == q)
                    {
                        //printf("%d -> %d\n", p, q);
                        found_neighbour = TRUE;
                        break;
                    }
                }

#ifdef DEBUG_IMPASS
                if ((!found_neighbour) && (opt.verbose))
                {
                    fprintf(stderr, "Impassable border found %d => %d\n", p, q);
                }
#endif

                /* Mark pixel as a border */
                if (!provs[p].sea || !provs[q].sea)
                {
                    border_type = LAND_BORDER;
                }
                else
                {
                    border_type = SEA_BORDER;
                }

                /* Use land border */
                map[y][x] = border_type;
                if (!found_neighbour && (opt.impass > 0.0))
                {
                    if (y > 0) map[y-1][x] = border_type;
                    if (x > 0) map[y][x-1] = border_type;
                    if ((x > 0) && (y > 0)) map[y-1][x-1] = border_type;
                    if (opt.triple_thick)
                    {
                        if (y > 1) map[y-2][x] = border_type;
                        if (x > 1) map[y][x-2] = border_type;
                    }
                }

            }
        }
    }
}

/*
* Place capitals and draw borders on the given map.
*/
void create_provinces(void)
{
    int i;
    int x;
    int y;

    /* Make temp map array */
    make_temp_map();

    /* Check verbosity */
    if (opt.verbose)
    {
        /* Message */
        printf("Placing province capitals...\n");
    }

    /* Place capitals */
    place_capitals();

    /* Collect list of capitals */
    collect_capitals();

    /* Check verbosity */
    if (opt.verbose)
    {
        /* Message */
        printf("Spreading provinces out from capitals...\n");
    }

    /* Spread provices out from the capitals */
    spread_provinces();

    /* Collect pixel numbers */
    for (y=0; y<opt.height; y++)
    {
        for (x=0; x<opt.width; x++)
        {
            if ((temp_map[y][x] > 0) && (map[y][x]>=0) &&
                (map[y][x]<TERRAIN_TYPE))
            {
                provs[temp_map[y][x]].pix[map[y][x]]++;
                provs[temp_map[y][x]].total_pixels++;
            }
        }
    }

    /* Check verbosity */
    if (opt.verbose && opt.smooth > 0)
    {
        /* Message */
        printf("Smoothing borders...\n");
    }

    /* Smooth borders as requested */
    for (i = 0; i < opt.smooth; i++)
    {
        /* Smooth province borders */
        smooth_borders();
    }

    /* Clean up shorelines if necessary */
    if (opt.smooth && !opt.no_force_water) force_water();

    /* Collect province neighbour information */
    collect_neighbours();

    /* create some impassable borders */
    if (opt.impass > 0.0) create_impasses();

    /* Check verbosity */
    if (opt.verbose)
    {
        /* Message */
        printf("Drawing borders...\n");
    }

    /* Draw the borders */
    draw_borders();

    // Free up some memory we shouldn't need anymore
    // list is used for mostly temporary stuff, so we shouldn't miss it,
    // but temp_map has the per-pixel province locations, which I think we might 
    // someday want to use.
    free(list);
}
