/*
 * Dominions random map generator.
 *
 * Copyright (c) 2002 Keldon Jones
 *
 * Modified by Philippe Duchon (2004) for compatibility with Dominions 2
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "dommap.h"

/*
 * Return the pixels encountered in a line drawn from the two given
 * points.
 */
static int get_path(int *path, int y1, int x1, int y2, int x2)
{
	int y, x;
	int dy, dx;
	int ay, ax;
	int sy, sx;
	int half, full;
	int frac, m;
	int n = 0;

	/* Get deltas */
	dy = y2 - y1;
	dx = x2 - x1;

	/* Get absolute deltas */
	ay = (dy < 0) ? (-dy) : dy;
	ax = (dx < 0) ? (-dx) : dx;

	/* Get delta signs */
	sy = (dy < 0) ? -1 : 1;
	sx = (dx < 0) ? -1 : 1;

	/* Check for vertical line */
	if (!dy)
	{
		/* Get column */
		x = x1;

		/* Scan straight up */
		for (y = y1; y != y2; y += sy)
		{
			/* Grab pixels */
			path[n++] = map[y][x];
		}

		/* Done */
		return n;
	}

	/* Check for horizontal line */
	if (!dx)
	{
		/* Get row */
		y = y1;

		/* Scan straight across */
		for (x = x1; x != x2; x += sx)
		{
			/* Grab pixels */
			path[n++] = map[y][x];
		}

		/* Done */
		return n;
	}

	/* Number of units in one half-pixel */
	half = ay * ax;

	/* Number of units in one full-pixel */
	full = half * 2;

	/* Mostly vertical */
	if (ay > ax)
	{
		/* Start at center of pixel */
		frac = 0;

		/* Compute slope */
		m = ax * ax * 2;

		/* Start column */
		x = x1;

		/* Loop until the end */
		for (y = y1; y != y2; y += sy)
		{
			/* Grab a pixel */
			path[n++] = map[y][x];

			/* Advance X-fraction */
			frac += m;

			/* Check for rollover */
			if (frac >= half)
			{
				/* Advance full step */
				x += sx;

				/* Start fraction over */
				frac -= full;
				
				/* Grab this pixel too */
				path[n++] = map[y][x];
			}
		}
	}

	/* Mostly horizontal */
	else
	{
		/* Start at center of pixel */
		frac = 0;

		/* Compute slope */
		m = ay * ay * 2;

		/* Start row */
		y = y1;

		/* Loop until the end */
		for (x = x1; x != x2; x += sx)
		{
			/* Grab a pixel */
			path[n++] = map[y][x];

			/* Advance Y-fraction */
			frac += m;

			/* Check for rollover */
			if (frac >= half)
			{
				/* Advance full step */
				y += sy;

				/* Start fraction over */
				frac -= full;
				
				/* Grab this pixel too */
				path[n++] = map[y][x];
			}
		}
	}

	/* Return length of path */
	return n;
}

/*
 * Return true if Dominions would consider the two given provinces
 * neighbours.
 *
 * This function is needed because we cannot simply dump every
 * neighbour relationship to the .map file.  Dominions doesn't seem
 * to be able to handle more than about 100.  So we need to not
 * dump superfluous #neighbour lines.  If Dominions would figure
 * out that they neighbour anyway, there's no need (and harm) in
 * doing so.
 *
 * In the future hopefully this function can be removed.  If
 * Illwinter will increase the possible #neighbour lines to, say,
 * 1000 or 2000 instead of 100, we should be alright.
 */
int dominions_neighbours(int p1, int p2)
{
	int y1, x1, y2, x2;
	int dy, dx;
	int path[800], path_len;
	int i, n = 0;
	int b1, b2;

	/* Get province capitals */
	y1 = provs[p1].cap_y;
	x1 = provs[p1].cap_x;
	y2 = provs[p2].cap_y;
	x2 = provs[p2].cap_x;

	/* Get deltas */
	dy = y1 - y2;
	dx = x1 - x2;

	/* Check for too far (400 pixels) */
	if (dy * dy + dx * dx > 400 * 400) return 0;
	
	/* Get path of line between capitals */
	path_len = get_path(path, y1, x1, y2, x2);

	/* Scan path looking for border crossings */
	for (i = 0; i < path_len - 1; i++)
	{
		/* Check for currently not in a border */
		b1 = (path[i] != LAND_BORDER && path[i] != SEA_BORDER);

		/* Check for next pixel in a border */
		b2 = (path[i + 1] == LAND_BORDER || path[i + 1] == SEA_BORDER);

		/* Check for border crossing */
		if (b1 && b2) n++;
	}

	/* Return true if only crossed one border */
	if (n < 2) return 1;

	/* Not obvious neighbours */
	return 0;
}
