import java.io.*;
import java.util.*;

public class DomIIBackup extends Thread {

  static final int BUFFSZ = 131072;
  static final byte[] fbuf = new byte[ BUFFSZ ];

  static void copy( File fin, File fout ) {
    InputStream in = null;
    OutputStream out = null;
    try {
      in = new FileInputStream( fin );
      out = new FileOutputStream( fout );
      while( true ) {
        synchronized( fbuf ) {
          int bytesRead = in.read( fbuf );
          if( bytesRead < 0 )
            break;
          out.write( fbuf, 0, bytesRead );
        }
      }
    } catch( FileNotFoundException e ) {
      e.printStackTrace();
    } catch( IOException e ) {
      e.printStackTrace();
    } finally {
      try {
        if( in != null )
          in.close();
        if( out != null )
          out.close();
      } catch( IOException e ) {
        e.printStackTrace();
      }
    }
  }

  class SaveFolderTracker extends Thread {

    class SaveFolder extends File implements Comparable {

      public SaveFolder( File f, String s ) {
        super( f, s );
        if( exists() ) {
          File[] turnFiles = listFiles( new FilenameFilter() {
              public boolean accept( File dir, String name ) {
                return name.endsWith( ".2h" ) || name.endsWith( ".trn" );
              }
            } );
          for( int n = 0; n < turnFiles.length; n++ ) {
            File file = turnFiles[ n ];
            if( file.getName().endsWith( ".2h" ) ) {
              mostRecent2H = Math.max( mostRecent2H, file.lastModified() );
            } else {
              mostRecentTRN = Math.max( mostRecentTRN, file.lastModified() );
            }
          }
        }
      }

      long timeStamp() {
        long time = 0L;
        File[] turnFiles = listFiles( new FilenameFilter() {
            public boolean accept( File dir, String name ) {
              return name.endsWith( ".trn" );
            }
          } );
        for( int n = 0; n < turnFiles.length; n++ ) {
          time = Math.max( time, turnFiles[ n ].lastModified() );
        }
        return time;
      }

      public int compareTo( Object o ) {
        // not pretty but we assume we won't compare
        // anything but SaveFolder instances anyway
        if( o == null )
          return 1;
        SaveFolder s = ( SaveFolder )o;
        boolean e1 = exists();
        boolean e2 = s.exists();
        if( e1 && e2 )
          return ( int )( timeStamp() - s.timeStamp() );
        return e1 ? 1 : e2 ? -1 : getName().compareTo( s.getName() );
      }

      void copyFiles() {
        File[] files = getParentFile().listFiles( new FileFilter() {
            public boolean accept( File pathname ) {
              return !pathname.isDirectory();
            }
          } );
        for( int n = 0; n < files.length; n++ )
          copy( files[ n ], new File( this, files[ n ].getName() ) );
      }

      void save() {
        saveFolders.remove( this );
        System.out.println( SaveFolderTracker.this.tracker.getName() + ": saving new turn to " + getName() );
        if( !exists() )
          mkdir();
        copyFiles();
        saveFolders.add( this );
      }

      void update() {
        System.out.println( SaveFolderTracker.this.tracker.getName() + ": updating " + getName() );
        copyFiles();
      }
    }

    File tracker;
    TreeSet saveFolders;
    long mostRecentTRN = 0L;
    long mostRecent2H = 0L;
    boolean enabled = true;

    SaveFolderTracker( String name ) {
      tracker = new File( domDir, name );
      System.out.println( "tracking: " + tracker.getName() );
      saveFolders = new TreeSet();
      for( int n = 0; n < backups; n++ ) {
        StringBuffer buffer = new StringBuffer( Integer.toString( n ) );
        while( buffer.length() < 3 )
          buffer.insert( 0, '0' );
        saveFolders.add( new SaveFolder( tracker, buffer.toString() ) );
      }
    }

    public void run() {
      while( true ) {
        long t = mostRecentTRN;
        long h = mostRecent2H;
        File[] turnFiles = tracker.listFiles( new FilenameFilter() {
            public boolean accept( File dir, String name ) {
              return name.endsWith( ".2h" ) || name.endsWith( ".trn" );
            }
          } );
        for( int n = 0; n < turnFiles.length; n++ ) {
          if( turnFiles[ n ].getName().endsWith( ".2h" ) ) {
            h = Math.max( h, turnFiles[ n ].lastModified() );
          } else {
            t = Math.max( t, turnFiles[ n ].lastModified() );
          }
        }
        if( t > mostRecentTRN ) {
          ( ( SaveFolder )saveFolders.first() ).save();
        } else if( h > mostRecent2H ) {
          ( ( SaveFolder )saveFolders.last() ).update();
        }
        mostRecentTRN = t;
        mostRecent2H = h;
        try {
          sleep( sleep );
        } catch( InterruptedException e ) {}
      }
    }
  }

  File domDir;
  HashMap trackers = new HashMap();
  HashSet excludeDirs = new HashSet();
  long sleep;
  int backups;

  public DomIIBackup( Properties properties ) {
    domDir = new File( properties.getProperty( "d2backup.d2home" ) );
    backups = Integer.parseInt( properties.getProperty( "d2backup.backups" ) );
    sleep = Integer.parseInt( properties.getProperty( "d2backup.sleep" ) ) * 1000;
    StringTokenizer tokenizer = new StringTokenizer( properties.getProperty( "d2backup.exclude" ), ":" );
    while( tokenizer.hasMoreElements() )
      excludeDirs.add( tokenizer.nextToken() );
  }

  void addSaveFolderTracker( String dirName ) {
    if( trackers.containsKey( dirName ) )
      return;
    SaveFolderTracker tracker = new SaveFolderTracker( dirName );
    trackers.put( dirName, tracker );
    tracker.start();
  }

  public void run() {
    while( true ) {
      File[] domSubDirs = domDir.listFiles( new FileFilter() {
          public boolean accept( File pathname ) {
            if( !pathname.isDirectory() )
              return false;
            if( excludeDirs.contains( pathname.getName() ) )
              return false;
            File[] domSaveDirs = pathname.listFiles( new FilenameFilter() {
                public boolean accept( File dir, String name ) {
                  return name.endsWith( ".trn" );
                }
              } );
            return domSaveDirs.length > 0;
          }
        } );
      for( int n = 0; n < domSubDirs.length; n++ )
        addSaveFolderTracker( domSubDirs[ n ].getName() );
      try {
        sleep( sleep );
      } catch( InterruptedException e ) {}
    }
  }

  public static void main( String[] args ) {
    Properties d2props = new Properties();
    d2props.setProperty( "d2backup.d2home", System.getProperty( "user.dir" ) );
    d2props.setProperty( "d2backup.sleep", "2" );
    d2props.setProperty( "d2backup.backups", "10" );
    d2props.setProperty( "d2backup.exclude", "doc:maps:mods:newlords:rawsound" );
    try {
      d2props.load( new FileInputStream( "domIIbackup.props" ) );
    } catch( FileNotFoundException e ) {
    } catch( IOException e ) {
      e.printStackTrace();
    }
    d2props.list( System.out );
    new DomIIBackup( d2props ).start();
  }
}
