#!/usr/bin/perl
package WeightedSets;
%namedSet = ();
$new_line = "\n";

sub clearSets {
  %namedSet = ( );
}

sub printSet {
  my (%setHash) = @_;
  foreach $element (sort keys %setHash) {
    print "$element => " . $setHash{ $element } . $new_line;
  }
}

sub parseSet {
  # The incoming set description
  $set_description = shift(@_);

  # The resulting set
  my %parsedHash = ();

  $set_description =~ 
    /\s*\{        # A { preceded by any amount of whitespace
  (.*)            # The contents of the { } brackets
    \}\s*         # A } followed by any amount of whitespace
      (<x\s*(\d+)\s*>)?/x ;  # Zero or one <xN> multipliers, where N is one or more digits
  
  $set_contents = $1;
  # If no multiplicator is set, the default is 1
  $set_multiplier = $3 ? $3 : 1;

  # Split the elements apart
  @set_elements = split /,/, $set_contents;

  # Process every element
  foreach $element (@set_elements) {
    $element =~ 
      /\s*(\d+)\s*   # An integer number possibly surrounded by whitespace
      -? # An optional range mark
      \s*(\d+)?\s*  # An optional second number, possibly surrounded by whitespace
      (<x\s*(\d+)\s*>)?   # An optional multiplier bracket <xN> where N is one or more digits
	/x; # End of the regexp

    $range_start = $1;
    $range_end = $2 ? $2 : $1;
    $range_multiplier = $4 ? $4 : 1;

    for( $i = $range_start; $i <= $range_end; ++$i ) {
      $parsedHash{$i} = $parsedHash{$i} + $range_multiplier * $set_multiplier;
      $total_weights = $total_weights + $range_multiplier * $set_multiplier;
    } # end for -- cover the range of the element
  } # end foreach -- all elements in the set

  return %parsedHash;
} # end sub parseSet

sub countSetWeights {
  my (%setHash) = @_;
  my $total_weights = 0;
  foreach $element (sort keys %setHash) {
    $total_weights = $total_weights + $setHash{ $element };
  }

  return $total_weights;
} # end sub countSetWeights

sub setUnion {
  my ($aRef, $bRef) = @_;
  my %resultSet = ( );

  %setA = %$aRef;
  %setB = %$bRef;

  foreach $element (keys %setA) {
    $resultSet{ $element } = $setA{ $element };
  }

  foreach $element (keys %setB) {
    $resultSet{ $element } = $resultSet{ $element } + $setB{ $element };
  }

  return %resultSet;
} # end sub setUnion

sub setDifference {
  my ($aRef, $bRef) = @_;
  my %resultSet = ( );
  %setA = %$aRef;
  %setB = %$bRef;

  foreach $element (keys %setA) {
    if( ! ($setB{ $element }) ) {
      $resultSet{ $element} = $setA{ $element };
    } # end else - this element is not in B, so copy over
  } # end if -- for every element in setA

  return %resultSet;
} # end sub setDifference

sub setIntersect {
  my ($aRef, $bRef) = @_;
  my %resultSet = ( );
  %setA = %$aRef;
  %setB = %$bRef;

  foreach $element (keys %setA) {
    if( $setB{ $element } ) {
      $valueA = $setA{ $element };
      $valueB = $setB{ $element };

      # add the minimum weight
      $resultSet{ $element } = $valueA < $valueB ? $valueA : $valueB;
    } # end if - this element also occurs in setB
  } # end if -- for every element in setA

  return %resultSet;
} # end sub setIntersect

sub parseSetExpression {
  # The incoming set description
  my $remaining = shift(@_);
  my $operator = "";
  # The resulting set
  my %parsedHash = ();
  my %nextSet = ();

  if( $remaining =~ /
		     (\s*({         # The start of a set
		     (\s*\d+\s*   # An integer number possibly surrounded by whitespace
		     (-\s*\d+)?\s*  # An optional range mark and second number, possibly surrounded by whitespace
		     (<x\s*\d+\s*>)?\s*   # An optional multiplier bracket
		     ,?\s*)+            # An optional comma
		   })|(\w+)\s*          # The end of a set
		     (<x\s*\d+\s*>)?) # An optional  multiplier bracket such as <x 12 > or <x2>
		     (.*)/x # Followed by the rest of the expression
    ) {
    if( $2 ) {
      %parsedHash = parseSet( $2 );
    }
    else {
      %parsedHash = getNamedSet( $6 );
    }
    $remaining = $8;
  } 
  else {
    return;
  }

  while( $remaining =~ /\S./ ) {
    if( !($remaining =~    
	  /\s*([U|\^|\\])\s+ # One of the three set operators U, ^ or \
	   (({         # The start of a set
	   (\s*\d+\s*   # An integer number possibly surrounded by whitespace
	   (-\s*\d+)?\s*  # An optional range mark and second number, possibly surrounded by whitespace
	   (<x\s*\d+\s*>)?\s*   # An optional multiplier bracket
	   ,?\s*)+            # An optional comma
	 })|(\w+)\s*          # The end of a set
	   (<x\s*\d+\s*>)?) # An optional  multiplier bracket
      (.*)/x # Followed by the rest of the expression
	 )) {
      return;
    } # end if 
    $operator = $1;
    $remaining = $9;

    my $old_2 = $2;
    my $old_7 = $7;
    if( $7 =~ /\w+/ ) {
      %next_set = getNamedSet( $old_7 );
    }
    else {
      %next_set = parseSet( $old_2 );
    }

    if( $operator eq "U" ) {
      %parsedHash = setUnion( \%parsedHash, \%next_set );
    }
    elsif( $operator eq "^" ) {
      %parsedHash = setIntersect( \%parsedHash, \%next_set );
    }
    elsif( $operator eq "\\" ) {
      %parsedHash = setDifference( \%parsedHash, \%next_set );
    } 
    else {
      print "ERROR. Unknown operator [$operator]" . $new_line;
      print "Remaining: $remaining " . $new_line;
    }
  } # end while $remaining contains at least one non-whitespace
    # character

  return %parsedHash;
} # end sub parseSetExpression

# Called with two parameters - the name of the set and the set
# description
sub parseNamedSetExpression {
  # The incoming set description
  my $setName = shift(@_);
  my $setDescription = shift (@_);
  my %set = parseSetExpression $setDescription;
  $namedSet{ $setName } = \%set;

  return %set;
} # end sub parseNamedSet

# Called with the name of a set, and if that set is known returns the
# set
sub getNamedSet {
  my $setName = shift(@_);
  my $setRef = $namedSet{ $setName };

  if( $setRef ) {
    return %$setRef;
  }
  else {
    return 0;
  }
} # end sub getNamedSet

# Chooses a random element from a weighted set, where the higher
# weighted elemnts have a greater chance to be selected.
#
# Called with two arguments, first the total weight of the set
# then the set itself
sub chooseRandomElement {
  my (%weightedSet) = @_;
  my $totalWeight = countSetWeights( %weightedSet );

  my $total = int( rand( $totalWeight) );
  my $element;
  
  foreach $element (keys %weightedSet ) {
    $total = $total - $weightedSet{ $element };
    if( $total < 0 ) {
      return $element;
    }
  } # end foreach -- find the element which tips us over the limit

  return $element;
} # end sub chooseRandomElement


1;
