#!/usr/bin/env python

import os, stat, time, re
import wx
import ConfigParser
import tarfile
import threading
import win32file
import win32con


class DomBackup(threading.Thread):
    
    def __init__(self, threadNum, window):
        threading.Thread.__init__(self)
        self.threadNum = threadNum
        self.window = window
        self.timeToQuit = threading.Event()
        self.timeToQuit.clear()
        
        cfg = ConfigParser.ConfigParser()
        cfg.read('config.cfg')
        
        self.dom3Dir = cfg.get('Main', 'dom3_directory')
        self.sourceDir = cfg.get('Main', 'watch_directory')
        self.targetDir = cfg.get('Main', 'save_directory')
        self.r_fileList = cfg.get('Main', 'r_file_collect').split()
        self.saveDelay = int(cfg.get('Main', 'save_delay'))
    
        msg = "Preparing...\n"
        wx.CallAfter(self.window.LogMessage, msg)     


    def stop(self):
        self.timeToQuit.set()


    def run(self):
        fileTime = 0
        fileTimeOld = 0
        msg = "Dom3Bak is now monitoring:\n\t%s\n" % self.sourceDir
        wx.CallAfter(self.window.LogMessage, msg)
   
        hDir = win32file.CreateFile (
            self.sourceDir,
            win32con.GENERIC_READ,
            win32con.FILE_SHARE_READ | win32con.FILE_SHARE_WRITE,
            None,
            win32con.OPEN_EXISTING,
            win32con.FILE_FLAG_BACKUP_SEMANTICS,
            None
        )
    
        while True:
            # Watch the target directory and sub directories
            results = win32file.ReadDirectoryChangesW (
                hDir,
                1024,
                True,
                win32con.FILE_NOTIFY_CHANGE_LAST_WRITE,
                None,
                None
            )
    
            for action, file in results:
                if self.timeToQuit.isSet():
                    break
                if file.endswith(".trn"):
                    full_filename = os.path.join(self.sourceDir, file)
                    fileTime = os.stat(full_filename)[stat.ST_MTIME]
                    if fileTime != fileTimeOld:
                        time.sleep(self.saveDelay)
                        fileTimeOld = fileTime
                        gameDir = file.split(os.sep)[0]
                        backupDir = os.path.join(self.targetDir, gameDir)
                        gameSetupFlag = False
                        # Make backup directory if none exists
                        if not os.path.exists(backupDir):
                            gameSetupFlag = True                       
                            os.makedirs(backupDir)
                            msg = "Creating new directory for %s\n" % gameDir
                            wx.CallAfter(self.window.LogMessage, msg)
                            msg = self.makeTar(gameDir, backupDir, "Game Setup", gameSetupFlag)
                            wx.CallAfter(self.window.LogMessage, msg)
        
                        # Else create tar file
                        else:
                            turnNum = "Turn %04d" %(int(self.domTurn(gameDir)))
                            msg = self.makeTar(gameDir, backupDir, turnNum, gameSetupFlag)
                            wx.CallAfter(self.window.LogMessage, msg)

        
    def domTurn(self, gameName):
        gamePath = os.path.join(self.dom3Dir, "savedgames", gameName)
        # Read turn number from .trn file
        for file in os.listdir(gamePath):
            if file.endswith(".trn"):
                filePath = os.path.join(gamePath, file)
                f = open(filePath, 'rb')
                content = f.read(15)
                gameTurn = (ord(content[13]) * 256) + ord(content[14])
                f.close()
        # return as number.
        return gameTurn
    

    def makeTar(self, gameDir, backupDir, tarName, allFilesFlag):
        zfilename = backupDir + os.sep + tarName + ".tar.gz"
        zout = tarfile.open(zfilename, "w:gz")

        # Place files in tar file.
        for f in os.listdir(os.path.join(self.sourceDir, gameDir)):
            if allFilesFlag:
                zout.add(os.path.join(self.sourceDir, gameDir, f).encode("latin-1"))
                wx.CallAfter(self.window.PopulateGameChoice)

            else:
                for r_item in self.r_fileList:
                    if f.endswith(r_item):
                        zout.add(os.path.join(self.sourceDir, gameDir, f).encode("latin-1"))
                        wx.CallAfter(self.window.PopulateTurnList)
        zout.close()

        # test if the file is a valid tar.gz file and print result
        if tarfile.is_tarfile(zfilename):
            msg = "Saved-->  %s\n" % tarName
        else:
            msg = "Error-->  %s is not valid\n" % zfilename
        return msg


class DirTools():
    
    def __init__(self):
        cfg = ConfigParser.ConfigParser()
        cfg.read('config.cfg')
        
        self.dom3Dir = cfg.get('Main', 'dom3_directory')
        self.sourceDir = cfg.get('Main', 'watch_directory')
        self.targetDir = cfg.get('Main', 'save_directory')


    def getGameList(self):
        gameSavePath = os.listdir(self.targetDir)
        return gameSavePath


    def getTurnFiles(self, gameDir):
        turnFileList = os.listdir(os.path.join(self.targetDir, gameDir))
        return turnFileList
    

    def deleteFile(self, gameDir, turnFile):
        deleteFile = os.path.join(self.targetDir, gameDir, turnFile)
        os.remove(deleteFile)
        return True
        

    def restoreFile(self, gameDir, turnFile):
        restoreFile = os.path.join(self.targetDir, gameDir, turnFile)
        extractDir = os.path.splitdrive(self.sourceDir)[0]
        extractDir = os.path.join(extractDir, os.sep)
        tar = tarfile.open(restoreFile)
        tar.extractall(extractDir)
        tar.close()
        return True
    
    def deleteGameBackups(self, game):
        deletePath = os.path.join(self.targetDir, game)
        for file in os.listdir(deletePath):
            if file.endswith(".tar.gz"): #Only delete the backup files
                os.remove(os.path.join(deletePath, file))
        os.rmdir(deletePath)
        return True
    
    
