-- Monster combat

-- Load the combat monster subsystem which defines some basic stuff for all monster combat subsystems
load_subsystem("combat_monster")

-- Make the namespace
combat_monster.default = {}

-- Clearing project flags
-- clears the project flags used to apply damage.
combat_monster.__project_flags = 0

-- Define the monster combat itself
constant("COMBAT_MONSTER", combat_monster.register
{
	name = "Default monster combat"
	energy	= function() return get_player_energy(SPEED_WEAPON) end,
	attack = function(y, x, c_ptr, m_idx, m_ptr, t_idx, t_ptr, t_flags) combat_monster.attack(y, x, c_ptr, m_idx, m_ptr, t_idx, t_ptr, t_flags) end,
})

function combat_monster.attack(y, x, c_ptr, m_idx, m_ptr, t_idx, t_ptr, t_flags)
		-- declare our local variables and tables
		local params = {}
		local special = {}

		-- get the monster
		local monst = monster(m_idx)

		-- get armor
		-- find out what the monster is attacking.  if it is the player
		-- than get player armor.  otherwise it must be another monster,
		-- so get its armor.
		local ac
		if t_ptr == player then
			ac = player.ac
		else
			ac = t_ptr.ac
		end

		-- Get the monster name (or "it").
		local m_name = combat_monster.desc_self(monst)
		local t_name = combat_monster.desc_target(t_ptr)

		-- a loop that moves through all blows
		for ap_cnt = 0, flag_max_key(monst.blow) do

			-- declare visible and obviousness
			-- I'm not entirely sure why these are really necessary
			local visible = false
			local obvious = false

			-- reset the damage value
			params.dam = 0

			-- Get the monster attack
			local blow = flag_get_blow(monst.blow, ap_cnt)
			local method = __monster_attack_methods[blow.method]
			local effect = __monster_attack_types[blow.effect]

			-- no more attacks
			if not blow then break end
			if (blow.method == 0) then break end

			-- Visibility
			if (monst.ml) then visible = true end

			-- this assigns all blow information to a table called 'params',
			-- which in turn is called in the functions for determing attacks
			-- and damage
			params.m_name	= m_name
			params.t_ptr	= t_ptr
			params.t_name	= t_name
			params.t_flags	= t_flags
			params.effect	= effect
			params.method	= method
			params.ac		= ac
			params.blow		= blow


			-- This determines if the monster has gone through all its blows
			-- and is finished attacking.  I'm not entirely sure how it works,
			-- but it does.
			local done = hook.process(hook.HANDLE_MONSTER_BLOW, params)

			if not done then
			-- The monster has additional attacks

				if method.fct then
				-- do the attack method (see monster attack methods below)

					-- Hack -- set the distance for missiles
					local dist = distance(monst.fy, monst.fx, y, x)

					-- Hack -- test for hits by looking for c_ptr
					c_ptr = method.fct(y, x, dist, params, special, m_idx)

					if c_ptr ~= nil then
					-- the attack hit, now disturb and do damage

						-- Message
						combat_monster.action_message(monst, m_name, t_ptr, t_name, method.action)

						-- Disturb others
						-- for now this stays, but the Dominions system may rework it.
						if game_options.disturb_other and (monst.ml or t_ptr == player or t_ptr.ml) then
							term.disturb(1, 0)
						end

						if effect.fct then
						-- calculate the effect (see monster attack effects below)
						
							-- obviousness
							obvious = effect.obvious

							effect.fct(y, x, params, special, m_idx, c_ptr)
						end

					else
					-- It didn't hit, but it still disturbs.

						-- Visible monsters
						-- what's with visible monsters?
						if (monst.ml) then

							-- Disturbing
							term.disturb(1, 0)

							-- Message
							monster_player_msg(strcap(m_name).." misses "..t_name..".", (t_ptr == player) or combat_monster.__force_name, t_ptr.ml)
						end
					end
				end


				-- The following essentially moves to the next blow for those
				-- monsters with multiple blows
				-- Analyze "visible" monsters only.
				-- Why do we analyze visible monsters only?
				if visible == true and (t_ptr == player or t_ptr.ml) then
					local blowmem = deep_get{memory.get_entry(race_info(monst),monst.ego),RT_BLOWS}
					if not blowmem[ap_cnt] then
						blowmem[ap_cnt] = 0
					end
					-- Count "obvious" attacks (and ones that cause damage)
					-- Not entirely sure what obviousness does, or if it is necessary
					if (obvious == true) or (params.dam > 0) or (blowmem[ap_cnt] > 10) then
						-- Count attacks of this type
						if (blowmem[ap_cnt] < 255) then
							blowmem[ap_cnt] = blowmem[ap_cnt] + 1
						end
					end
				end
			-- If the monster is done with all attacks, the function jumps
			-- to here and no attacking is done.
			end

			-- Don't keep hitting if the player is dead
			if (t_ptr == player) and (player.chp() < 0) then return end
		end
		-- i'm not entirely sure why this is necessary.
		return false
	end

-- All the monster attack methods
-- These are HOW the monster attacks.  eg melee or missile
add_monster_attack_method
{
	["name"]	= "HIT",
	["desc"]	= "hit",
	["action"]	= "hits @target@",
	["fct"]		= function(y, x, dist, params, special, m_idx, c_ptr)
		-- note qualities of the attack
		special.touched = true

		-- Get the monster
		local monst = monster(m_idx)

		-- Increase monster fatigue
		monst.flags[FLAG_MONST_FAT] = monst.flags[FLAG_MONST_FAT] + monst.flags[FLAG_MONST_ENCUM]

		-- Do DRN rolls for attack and defense
		local attroll = rng.drn()
		local defroll = rng.drn()

		-- Add attack and defense skills to the rolls
		attroll = attroll + monst.flags[FLAG_MONST_ATT]
		defroll = defroll + player.stat(A_DEFENSE)

		-- Subtract fatigue penalties from attack and defense rolls
		attroll = attroll - (monst.flags[FLAG_MONST_FAT] / 20)
		defroll = defroll - (player.cfat() / 10)

		if attroll > defroll then
		-- it hit

			c_ptr = cave(y, x)

			-- Hack -- return the c_ptr
			return c_ptr
		else
		-- it missed

			-- return nil to skip the effect function
			return nil
		end
	end,
}
add_monster_attack_method
{
	["name"]	= "SHOOT",
	["desc"]	= "shoot",
	["action"]	= "shoots @target@",
	["fct"]		= function(y, x, dist, params, special, m_idx, c_ptr)
		-- note the qualities of the attack
		special.shoot = true

		-- get the monster
		local monst = monster(m_idx)

		if monst.flags[FLAG_MONST_AMMO_CURRENT] > 0 then

			-- deplete ammo
			monst.flags[FLAG_MONST_AMMO_CURRENT] = monst.flags[FLAG_MONST_AMMO_CURRENT] - 1

			-- Check for deviation.
			-- Compute where a missile will land in case of deviation.
			if dist > ((monst.flags[FLAG_MONST_PREC] / 2) - 2) then
				local dev = (dist * (5 / 4)) / monst.flags[FLAG_MONST_PREC]
				local vy = rng(-dev, dev)
				local vx = rng(-dev, dev)
				local cov = 0

				-- get diagonal steps
				-- don't divide by zero
				if vy < 0 and vx < 0 then
					if vy <= vx then
						cov = abs(vy) / abs(vx)
					else
						cov = abs(vx) / abs(vy)
					end
				else
					cov = 0
				end

				-- modify where the missile was dropped by circularizing
				-- the area
				local div = (cov * (1 / 2)) + 1
				vy = vy / div
				vx = vx / div
				y = y + vy
				x = x + vx
			end

			c_ptr = cave(y, x)
			
			if c_ptr == player or c_ptr.m_idx > 0 then
			-- the missile hit a player/monster

				-- Hack -- return the c_ptr
				return c_ptr

			else
			-- It missed, but still strikes the ground

				local damroll = rng.drn()
				local protroll = rng.drn()

				damroll = damroll
				protroll = protroll

				params.dam = damroll - protroll

				if params.dam > 0 then

					project(m_idx, 0, y, x, params.dam, params.effect.type, PROJECT_KILL | PROJECT_STOP | PROJECT_NO_REFLECT | combat_monster.__project_flags)

				end

				-- return nil to skip the effect function
				return nil
			end
		end
	end,
}

-- All monster attack types
-- These are the effects of a monster hit.  eg pure damage.
add_monster_attack_type
{
	["name"]	= "HURT",
	["desc"]	= "hurt",
	["power"]	= 0,
	["type"]	= dam.MELEE,
	["obvious"]	= true,
	["fct"]		= function(y, x, params, special, m_idx, c_ptr)
	-- all blow values are fed in through the params table, and must be
	-- suffixed with params.foo

		-- Check for a critical hit.
		-- find out what fatigue value we are using, the players or
		-- another monster's
		local fat
		if params.t_ptr == player then

			-- get the player's fatigue
			fat = player.cfat()
		else

			-- the target must be another monster, so get that monster's
			-- fatigue
			fat = t_ptr.flags[FLAG_MONST_FAT]
		end

		-- Do the critroll
		local critroll = rng.drn()
		if (critroll - (fat / 15 )) < 2 then

			-- It critical hit.  Halve target protection.
			params.ac = params.ac / 2
		end	

		-- Now do DRN rolls for damage and protection
		local damroll = rng.drn()
		local protroll = rng.drn()

		-- Add monster damage and player protection to rolls
		damroll = damroll + rng.roll(params.blow.d_dice, params.blow.d_side)
		protroll = protroll + params.ac

		-- Compare the rolls and see if damage was inflicted.
		params.dam = damroll - protroll

		if params.dam > 0 then
		-- It did damage

			-- Apply the damage
			project(m_idx, 0, y, x, params.dam, params.effect.type, PROJECT_KILL | PROJECT_STOP | PROJECT_NO_REFLECT | combat_monster.__project_flags)

		-- if dam < protection, the function jumps to here, skipping the
		-- damage application.  this prevents healing through 'negative damage'
		end
	end,
}
add_monster_attack_type
{
	["name"]	= "SHOT",
	["desc"]	= "shot",
	["power"]	= 0,
	["type"]	= dam.MISSILE,
	["obvious"]	= true,
	["fct"]		= function(y, x, params, special, m_idx, c_ptr)
	-- all blow values are fed in through the params table, and must be
	-- suffixed with params.foo

		-- do attack and defense DRN rolls
		local attroll = rng.drn()
		local defroll = rng.drn()

		-- Compare attack and defense rolls, and see if a hit occurs
		if attroll > defroll then
		-- It hit

			-- Now do DRN rolls for damage and protection
			local damroll = rng.drn()
			local protroll = rng.drn()

			-- Add monster damage and player protection to rolls
			damroll = damroll + rng.roll(params.blow.d_dice, params.blow.d_side)
			protroll = protroll + params.ac

			-- Compare the rolls and see if damage was inflicted.
			params.dam = damroll - protroll

			if params.dam > 0 then
			-- It did damage

				-- Apply the damage
				project(m_idx, 0, y, x, params.dam, params.effect.type, PROJECT_KILL | PROJECT_STOP | PROJECT_NO_REFLECT | combat_monster.__project_flags)

			-- if dam < protection, the function jumps to here, skipping the
			-- damage application.  this prevents healing through 'negative damage'
			end

		-- attack roll failed, function jumps to here, and no damage is
		-- computed
		end
	end,
}

function default_monster_drop(m_idx, m_ptr, r_ptr)
	-- No drops
end