using System;
using System.Collections;
using System.Net;
using System.Threading; 

using Microsoft.DirectX;
using Microsoft.DirectX.DirectPlay;

using LobbyServer;
using SI_MiddleTier;


/// <summary>
/// Summary description for LobbyServer.
/// </summary>
public class Base_LobbyServer
{
	#region enums
	private enum challengedPlayerStatus
	{
		PlayerNotLoggedOn, //the player isn't logged on.
		AlreadyChallenged, //the player is already being challenged by someone else
		AlreadyChallenging, //the player is already challenging someone else
		OkayToChallenge //This player is a valid challenge target
	}
	#endregion

	#region Constants
	//public const string LobbyServerAddress="216.231.46.4";
	public const string LobbyServerAddress="127.0.0.1";
	public const int LobbyServerPort=1337;
	//public const int LobbyServerPort=23;
	public const string LobbyServerGuid = "{CF102364-2677-4a22-9009-BC86CF368A06}";
	//private const string _dbServerName = "mysql2.webcontrolcenter.com";	//"jones.theigl.com";
	private const string _dbServerName = "69.56.133.54";	//"jones.theigl.com";
	private const string _dbName = "lol";	//"lol2";

	//This is for the GameServer, so this LobbyServer can connect to it (for current player lists)
	public const string GameServerAddress="127.0.0.1";
	public const int GameServerPort=21337;
	public const string GameServerGuid = "{CF102364-2677-4a22-9009-BC86CF368A07}";
	
	public const int PacketTimoutDuration=60;        
	#endregion

	#region Private Member Variables
	private Server _lobbyServer;
	private MiddleTier _dbquery = null;
	public PlayerList _playerList = new PlayerList();
	public PlayerList _playerListInGameServer = new PlayerList();
	private int _gameTypeID = 1;	// Land of Legends, version 1.0  <-- this will always be a released version number

	private Client _gameServerClient = null;
	private bool _tryingToConnectToGameServer = false;
	private bool _waitingForResponse = false;
	private bool _connectedToGameServer = false;
	private bool _inLoop = false;
	#endregion

	#region Constructor
	public Base_LobbyServer()
	{
		// Do nothing
	}

	public Base_LobbyServer(bool doStartServer)
	{
		if (doStartServer)
		{
			StartServer();
		}
	}
	#endregion

	#region Destructor
	/// <summary>
	/// Clean up any resources being used.
	/// </summary>
	public virtual void Dispose()
	{
		if (_dbquery != null)
		{
			_dbquery.Dispose();
		}
	}
	#endregion

	#region Public Functions
	public virtual void AddUser(PlayerObject playerToAdd)
	{
		_playerList.Players.Add(playerToAdd.Username,playerToAdd);
		//PlayerListBox.Items.Add(playerToAdd.Username);
	}
	public virtual PlayerObject RemoveUser(int playerID)
	{
		PlayerObject __tempPlayer;
		foreach (string __username in _playerList.Players.Keys) 
		{
			__tempPlayer=(PlayerObject) _playerList.Players[__username];
			if (__tempPlayer.PlayerID==playerID)
			{
				_playerList.Players.Remove(__username);
				//PlayerListBox.Items.Remove(__tempPlayer.Username);
				return __tempPlayer;
			}
		}

		return null;
	}
	public void SendUserListToClient(int playerIDToUpdate)
	{
		NetworkPacket __returnPacket = new NetworkPacket();
		__returnPacket.Write(LobbyMessageType.CurrentUserList);
		PlayerObject __tempPlayer;
		foreach (string __key in _playerList.Players.Keys) 
		{
			__tempPlayer=(PlayerObject) _playerList.Players[__key];
			__returnPacket.Write(__tempPlayer.PlayerID); 
			__returnPacket.Write(__tempPlayer.Username); 
			//			__returnPacket.Write(__tempPlayer.PlayerRating); 
			//			__returnPacket.Write(__tempPlayer.PlayerRank); 
			__returnPacket.Write(__tempPlayer.Wins); 
			__returnPacket.Write(__tempPlayer.Losses); 
			__returnPacket.Write(__tempPlayer.XP); 
			__returnPacket.Write(__tempPlayer.Level); 
			__returnPacket.Write(__tempPlayer.IsRegisteredUser); 
			__returnPacket.Write(__tempPlayer.PreferencesString); 
		}
		_sendMessageToClient(playerIDToUpdate, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
		__returnPacket.Dispose();
	}
	public void StartServer () 
	{
		try
		{
			_lobbyServer = new Server(); //create a new server object

			Address __address = new Address(); // create an address which will indicate a serive provider type and a port for your app 

			__address.ServiceProvider = Address.ServiceProviderTcpIp;
			__address.AddComponent(Address.KeyPort, LobbyServerPort);

			ApplicationDescription __appDescription = new ApplicationDescription(); //describe your application to any client trying to connect 
	        
			__appDescription.Flags = SessionFlags.ClientServer | SessionFlags.NoDpnServer;
			__appDescription.GuidApplication=new Guid(LobbyServerGuid); //TODO: Insert your app GUID here 
			__appDescription.SessionName="Session Description";
	        
			try
			{
				_lobbyServer.Host(__appDescription, __address);
			}
			catch(InvalidDeviceAddressException)
			{
				//MessageBox.Show("Another instance of this server with the same application Guid was found. This instance will now exit.");
				return;
			}
	        
			WriteTextToChatArea("Server has started.");

			//_lobbyServer.PlayerCreated += new PlayerCreatedEventHandler(PlayerJoin);
			//_lobbyServer.IndicateConnect += new IndicateConnectEventHandler(PlayerAttJoin);
			//_lobbyServer.FindHostQuery += new FindHostQueryEventHandler(ThisHostIsFound);
			_lobbyServer.Receive += new ReceiveEventHandler(DataReceive);
			_lobbyServer.PlayerDestroyed += new PlayerDestroyedEventHandler(PlayerLeft);
			//_lobbyServer.PlayerDestroyed += new PlayerDestroyedEventHandler(PlayerLeft);

			_connectDB();
		}
		catch (Exception e)
		{
			SI_MiddleTier.MiddleTier.RecordExceptionToErrorFile("Lobby Server", e, "Problem in StartServer()");  //make sure the name of the server is listed -- it will be recorded in the log file.

			throw e;
		}
	}

	private void _connectDB()
	{
		// Connect to database
		_dbquery = new MiddleTier("lol");	//lol2

		// Find the Land of Legends game type id if there is one
		int[] __typeIDs = null;
		string[] __typeNames = null;
		int __count = _dbquery.GetGameTypeIDs(out __typeIDs, out __typeNames);

		for (int i = 0; i < __count; i ++)
		{
			if (__typeNames[i].ToLower() == "land of legends")
			{
				_gameTypeID = __typeIDs[i];
				break;
			}
		}
	}

	public void HandleSynchForLoggedInUser(PlayerObject loggingInPlayer)
	{
		//add the user to the localized user list(s)
		AddUser(loggingInPlayer);

		//Tell the client that their login succeeded
		NetworkPacket __returnPacket = new NetworkPacket();
		__returnPacket.Write(LobbyMessageType.LoginSuccess);
		__returnPacket.Write(loggingInPlayer.UserID);
		__returnPacket.Write(loggingInPlayer.XP);
		_sendMessageToClient(loggingInPlayer.PlayerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
		__returnPacket.Dispose();

		//Send the client a list of users already logged in
		SendUserListToClient(loggingInPlayer.PlayerID);

		//echo a logic success message to the console
		WriteTextToChatArea(loggingInPlayer.Username + " successfully logged in...");

	}
	public virtual void KickPlayer(PlayerObject playerToKick)
	{
		//write to the server console that the player was kicked.
		WriteTextToChatArea(playerToKick.Username + " was booted...");

		DestroyPlayer(playerToKick, true);
	}
	private void _sendPlayerOffToGame(PlayerObject playerLeaving, bool sendLogoffMessageToOtherPlayers)
	{
		WriteTextToChatArea(playerLeaving.Username + " is leaving to start a game...");

		DestoryPlayerGoingOffToGame(playerLeaving, sendLogoffMessageToOtherPlayers);
	}
	public void DestoryPlayerGoingOffToGame(PlayerObject playerToDestroy, bool sendLogoffMessageToOtherPlayers) 
	{
		int __tempPlayerID = playerToDestroy.PlayerID;

		//local UI and list -- that's handled in the event
		RemoveUser(__tempPlayerID);
        
		//boot the player
		NetworkPacket __returnPacket = new NetworkPacket ();
		__returnPacket.Write(LobbyMessageType.ServerKick);
		_sendMessageToClient(__tempPlayerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
		__returnPacket.Dispose();

		//_lobbyServer.DestroyClient( __tempPlayerID,__returnPacket);

		//notify the other players
		if (sendLogoffMessageToOtherPlayers) 
		{
			_tellAllPlayersAboutUserLogoff(__tempPlayerID,playerToDestroy.Username);
		}
	}
	public void DestroyPlayer(PlayerObject playerToDestroy, bool sendLogoffMessageToOtherPlayers)
	{
		int __tempPlayerID = playerToDestroy.PlayerID;

		//local UI and list -- that's handled in the event
		RemoveUser(__tempPlayerID);
        
		//boot the player
		NetworkPacket __returnPacket = new NetworkPacket ();
		__returnPacket.Write(LobbyMessageType.ServerKick);
		_sendMessageToClient(__tempPlayerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
		__returnPacket.Dispose();

		_lobbyServer.DestroyClient( __tempPlayerID,__returnPacket);

		//notify the other players
		if (sendLogoffMessageToOtherPlayers) 
		{
			_tellAllPlayersAboutUserLogoff(__tempPlayerID,playerToDestroy.Username);
		}
	}
	#endregion

	#region Private Functions
	private string _returnExceptionStringFromException (Exception e)
	{
		string __errorMessageText = "";
		__errorMessageText += "Error Message:  " + e.Message + System.Environment.NewLine + System.Environment.NewLine;
		__errorMessageText += "Method:  " + e.TargetSite  + System.Environment.NewLine;
		__errorMessageText += "Object/App:  " + e.Source + System.Environment.NewLine + System.Environment.NewLine;
		//__errorMessageText += "HelpLink:  " + e.HelpLink + System.Environment.NewLine + System.Environment.NewLine;
		__errorMessageText += "Stack Trace:  " + System.Environment.NewLine + e.StackTrace;
		return __errorMessageText;
	}
	private void _logException(Exception e)
	{
		_logException(e, "");
	}
	private void _logException(Exception e, string errorInfo)
	{
		string __errorMessageText=_returnExceptionStringFromException(e);
		//we really should have some sort of exception-recording mechanism here.
		WriteTextToChatArea("*******  WE CAUGHT AN EXCEPTION!!!!  *******");
		WriteTextToChatArea(__errorMessageText + System.Environment.NewLine + System.Environment.NewLine);
		WriteTextToChatArea("MORE INFO:  " + errorInfo);
		WriteTextToChatArea("*************  END EXCEPTION  **************");

		SI_MiddleTier.MiddleTier.RecordExceptionToErrorFile("Lobby Server", e, errorInfo);  

	}
	private void _sendSystemMessageToAllClients(string _message) 
	{
		NetworkPacket __returnPacket = new NetworkPacket ();
		__returnPacket.Write(LobbyMessageType.SystemMessage);
		__returnPacket.Write(_message);
		_sendMessageToClient((int) PlayerID.AllPlayers, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
		__returnPacket.Dispose();
	}
	private void _sendMessageToClient(int playerIDOfClient, NetworkPacket packetToSend, int packetTimeoutDuration, SendFlags sendFlags) 
	{
		try 
		{
			_lobbyServer.SendTo(playerIDOfClient, packetToSend, packetTimeoutDuration, sendFlags);
		} 
		catch (Exception e)
		{
			_logException(e); 
		}
	}

	private PlayerObject _createPlayerObjectForPlayer(int playerID, string username, int userID, string playerPreferences, bool isUsingDemoClient)
	{
		//int __playerRank = 0;
		//int __playerRating = 0;
		int __wins=0;
		int __losses=0;
		int __xp=0;
		int __level=1;
		bool __isRegisteredUser=false;


		// Look up in the DB the ranking/rating for this player.
		// List of stats in descending order by entry date
		string[] __result = _dbquery.GetPlayerStats(userID);

		_dbquery.GetPlayerWinsAndLosses(userID, ref __wins, ref __losses);

		if ( (__result != null) && (__result.Length > 0) )
		{
			__xp = int.Parse(MiddleTier.GetPlayerStatEntry(__result, "xp"));

 		    //Random __randomGenerator=new Random();
			//__xp = __randomGenerator.Next(100);


			// Test code to show we can add player stats
			//_dbquery.AddPlayerStat(userID, "xp", Convert.ToString(__xp + 3));

			//TODO:  pull this from the DB.
			__isRegisteredUser=true;

			if (isUsingDemoClient) 
			{
				//if they're using a demo client, treat them as a demo user.
				__isRegisteredUser=false;
			}

			__level=PlayerObject.GetLevelFromXP(__xp);
			
			//__playerRank = int.Parse(MiddleTier.GetPlayerStatEntry(__result, "rank"));
			//__playerRating = int.Parse(MiddleTier.GetPlayerStatEntry(__result, "rating"));
		} 
		else 
		{
			//TODO: We failed here, but we currently don't do anything.  When I (Scott) tried to test, we were hitting this code path. 
			//Why?  And what should be do about it?
			// The reason why you see an empty result of player stats is that we are not writing data to the GameStat table

			//__wins=0;
			//__losses=0;
			__xp=0;
			__isRegisteredUser=true;
			if (isUsingDemoClient) 
			{
				//if they're using a demo client, treat them as a demo user.
				__isRegisteredUser=false;
			}

			__level=PlayerObject.GetLevelFromXP(__xp);
		}

		//PlayerObject __tempPlayer = new PlayerObject(playerID, username, 1000, 5, userID);
		PlayerObject __tempPlayer = new PlayerObject(playerID, username, userID, __wins, __losses, __xp, __level, __isRegisteredUser, playerPreferences);

		return __tempPlayer;
	}

	public virtual void WriteTextToChatArea(string textToWrite) 
	{
		// Allow inherited class to override
	}

	private void _tellAllPlayersAboutUserLogoff(int playerID, string username) 
	{
		NetworkPacket __returnPacket = new NetworkPacket ();
		__returnPacket.Write(LobbyMessageType.UserLogoff);
		__returnPacket.Write(playerID);
		__returnPacket.Write(username);
		_sendMessageToClient((int) PlayerID.AllPlayers, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
		__returnPacket.Dispose();
	}

	private challengedPlayerStatus _canPlayerAcceptChallenge(string challengedPlayerUsername) 
	{
		if (_playerList.Players.ContainsKey(challengedPlayerUsername)) 
		{
			PlayerObject __playerObject=(PlayerObject) _playerList.Players[challengedPlayerUsername];
			int __challengedPlayerID = __playerObject.PlayerID;

			if (ChallengeInfo.isPlayerAlreadyBeingChallenged(challengedPlayerUsername)) 
			{
				return challengedPlayerStatus.AlreadyChallenged;
			}

			if (ChallengeInfo.isPlayerAlreadyChallengingSomeone(challengedPlayerUsername)) 
			{
				return challengedPlayerStatus.AlreadyChallenging;
			}
			return challengedPlayerStatus.OkayToChallenge;
		} 
		else 
		{
			return challengedPlayerStatus.PlayerNotLoggedOn;
		}
	}
	private void _rejectChallenge_AlreadyChallenged(int challengingPlayerID, string challengedUserName) 
	{ 
		//user is already being challenged by someone else
		NetworkPacket __returnPacket = new NetworkPacket();
		__returnPacket.Write(LobbyMessageType.ChallengeAlreadyIssuedToPlayer);
		WriteTextToChatArea(_playerList.GetPlayerByID(challengingPlayerID).Username + " tried to challenge " + challengedUserName + " but that player is already being challenged.");
		_sendMessageToClient(challengingPlayerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
		__returnPacket.Dispose();
	}
	private void _rejectChallenge_AlreadyChallenging(int challengingPlayerID, string challengedUserName) 
	{ 
		//user is already challenging someone else
		NetworkPacket __returnPacket = new NetworkPacket();
		__returnPacket.Write(LobbyMessageType.ChallengeAlreadyIssuedToPlayer);
		WriteTextToChatArea(_playerList.GetPlayerByID(challengingPlayerID).Username + " tried to challenge " + challengedUserName + " but that player is already challenging someone else.");
		_sendMessageToClient(challengingPlayerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
		__returnPacket.Dispose();
	}
	private void _rejectChallenge_TargetNotLoggedIn(int challengingPlayerID, string challengedUserName) 
	{ 
		//We don't see the challenged user around.  Gracefully tell the user this.
		NetworkPacket __returnPacket = new NetworkPacket();
		__returnPacket.Write(LobbyMessageType.ChallengeTargetNotLoggedIn);
		WriteTextToChatArea(_playerList.GetPlayerByID(challengingPlayerID).Username + " tried to challenge " + challengedUserName + " + but they weren't logged in.");
		_sendMessageToClient(challengingPlayerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
		__returnPacket.Dispose();
	}

	private void _notifyPlayerOfCancelledChallenge(int playerID, ChallengeInfo challenge)
	{            
		//        if (_playerList.Players.ContainsKey(__username)) 
		//        {
		//            __playerObject=(PlayerObject) _playerList.Players[__username];
		//            __recivingPlayerID = __playerObject.PlayerID;
		//        }
		if (_playerList.PlayerIDExists(playerID)) 
		{
			NetworkPacket __returnPacket = new NetworkPacket();
			__returnPacket.Write(LobbyMessageType.ChallengeCancelled);
			_sendMessageToClient(playerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
			__returnPacket.Dispose();
		}
	}
	private void _cancelAnyOutstandingChallengesForPlayer(int playerID)
	{
		//make sure the player actually does exist.
		if (_playerList.PlayerIDExists(playerID)) 
		{
			PlayerObject __playerObject=(PlayerObject) _playerList.GetPlayerByID(playerID);
			PlayerObject __otherPlayerInChallege=null;
			ChallengeInfo __challenge = ChallengeInfo.GetChallengeForPlayer(__playerObject.Username);

			// clear all the challenges..
			while (__challenge!=null)
			{
				lock (ChallengeInfo.Challenges) 
				{
					//find out who the other player in the challenge is, and tell them the challenge is off.
					if (__challenge.IsPlayerChallengerInChallenge(__playerObject.Username))
					{
						__otherPlayerInChallege=__challenge.ChallengedPlayer;
					} 
					else 
					{
						__otherPlayerInChallege=__challenge.ChallengingPlayer;
					}

					//tell the __otherPlayerInChallege that the challenge is cancelled.
					_notifyPlayerOfCancelledChallenge(__otherPlayerInChallege.PlayerID,__challenge);
					//CHANGE:  NO NEED TO TELL CHALLENGING PLAYER (was: "try telling the cancelling player.  They might have just been disconnected though.")
					//_notifyPlayerOfCancelledChallenge(playerID,__challenge);

					//write a message about it
					WriteTextToChatArea("Challenge between " + __challenge.ChallengingPlayer.Username + " and " + __challenge.ChallengedPlayer.Username + " has been cancelled.");

					//nuke the challenge.
					ChallengeInfo.CancelChallenge(__challenge);
                        
					__challenge = ChallengeInfo.GetChallengeForPlayer(__playerObject.Username);

				}
			}
		}             
	}

	
	private void _connectToGameServerAndShakeHands() 
	{
		IPAddress __hostAddress = IPAddress.Parse(GameServerAddress);

		while (!_connectedToGameServer)
		{
			_connectToGameServer(__hostAddress, GameServerPort, GameServerGuid, 15f);

			if (!_connectedToGameServer) 
			{
				//the conneciton failed.  Wait 5 seconds a try again.
				WriteTextToChatArea("Connection to Game Server failed.  Waiting 5 seconds to try again.");
				System.Threading.Thread.Sleep(5000);
			} 
			else 
			{
				//we're connected.  Tell them what kind of user we are.
			}
		}
	}
	private void _connectToGameServer(IPAddress hostaddress, int GameServerPort, string GameServerGuid, double ServerConnectTimeout) 
	{
		WriteTextToChatArea("Attempting to connect to game server...");   
		_inLoop=true;
		_waitingForResponse=true;
		_tryingToConnectToGameServer=true;
		
		_gameServerClient= new Client(InitializeFlags.DisableParameterValidation); //create a new client
		_gameServerClient.ConnectComplete += new ConnectCompleteEventHandler(_connectComplete);
        
		Address __address = new Address();
		__address.ServiceProvider=Address.ServiceProviderTcpIp;
		Address __hostAddress = new Address(hostaddress);
		__hostAddress.ServiceProvider = Address.ServiceProviderTcpIp; //tell your app that the server will use TCPIP too
		__hostAddress.AddComponent(Address.KeyPort, GameServerPort); //specify the port at which your app will look for the server
		ApplicationDescription __desc = new ApplicationDescription();
		__desc.GuidApplication = new Guid(GameServerGuid); //TODO: Insert your app GUID here
		__desc.Flags = SessionFlags.ClientServer | SessionFlags.NoDpnServer; //we will use the same flags as in our server 

		DateTime __connectTimeoutTime = DateTime.Now.AddSeconds(ServerConnectTimeout);

		//_hConnect = new AutoResetEvent(false);

		try 
		{
			_gameServerClient.Connect(__desc, __hostAddress, __address, null, 0);
		}
		catch (Exception e)
		{
			WriteTextToChatArea("Encountered an exception on _gameServerClient.Connect(), and don't know why.");
			_inLoop=false;
			_waitingForResponse=false;
		}

		while (_waitingForResponse==true && _inLoop==true)
		{
			// Don't kill the CPU, allow other things to happen while we wait
			//Application.DoEvents();
			
			System.Threading.Thread.Sleep(100);
		}

		_gameServerClient.ConnectComplete -= new ConnectCompleteEventHandler(_connectComplete);

		_tryingToConnectToGameServer=false;
	}
	private void _connectComplete(object sender, ConnectCompleteEventArgs e)
	{
		if (e.Message.ResultCode != 0) 
		{
			//we didn't really connect
			_connectedToGameServer = false; //share the good news with our local variable! 

			//PUT MESSAGE IN EVENT LOG HERE
			WriteTextToChatArea("Unable to connect to the Game Server for some reason.");
		}
		else
		{
			//we seemingly did connect
			_connectedToGameServer = true;//share the good news with our local variable! 
			WriteTextToChatArea("Connected to Game Server.");
		}

		_gameServerClient.Receive += new ReceiveEventHandler(_dataReceiveFromGameServer);
		_gameServerClient.SessionTerminated += new SessionTerminatedEventHandler(_serverDisconnectFromGameServer);

		//either way, tell the connection loop that we are done trying to connect
		_waitingForResponse=false;

		//now we need to send the game server a InGamePlayerListLogin message to tell it we're interested in player lists.
		NetworkPacket __loginPacket = new NetworkPacket();
		__loginPacket.Write(GameMessageType.InGamePlayerListLogin);
		try 
		{
			_gameServerClient.Send(__loginPacket, 0, SendFlags.Guaranteed | SendFlags.NoLoopback);
			//_sendMessageToClient(__playerID, __loginPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);  
			__loginPacket.Dispose();                              
		} 
		catch {}
	}
	private void _serverDisconnectFromGameServer(object sender, SessionTerminatedEventArgs e)
	{
		//we got dropped from the server -- we should try to reconect.
		WriteTextToChatArea("Lost connection to Game Server.  About to launch reconnect attempt...");
		_gameServerClient.Receive -= new ReceiveEventHandler(_dataReceiveFromGameServer);
		_gameServerClient.SessionTerminated -= new SessionTerminatedEventHandler(_serverDisconnectFromGameServer);
		_connectToGameServerAndShakeHands();
	}        
	public void _dataReceiveFromGameServer(object sender, ReceiveEventArgs e)
	{
		// we need to process the packets here.  
		NetworkPacket __recievedPacket = e.Message.ReceiveData; //get the data sent to the server 
		GameMessageType __messageType = (GameMessageType)__recievedPacket.Read(typeof(GameMessageType));

		int  __sendingPlayer=0;
		int __playerID=0;
		string __username="";
		int __userID=0;
		int __wins=0;
		int __losses=0;
		int __xp=0;
		int __level=0;
		bool __isRegisteredUser=false;
		string __playerPreferences="";
		PlayerObject playerToAdd;

		switch (__messageType) 
		{
			case GameMessageType.CurrentInGamePlayerList:
				//_clearOutPlayerList();
				lock (_playerListInGameServer)
				{
					_playerListInGameServer=new PlayerList();
					while (__recievedPacket.Position<__recievedPacket.Length)
					{
						__playerID=(int) __recievedPacket.Read(typeof(int));
						__username=(string) __recievedPacket.ReadString();
						__userID=(int) __recievedPacket.Read(typeof(int));
						__wins=(int) __recievedPacket.Read(typeof(int));;
						__losses=(int) __recievedPacket.Read(typeof(int));;
						__xp=(int) __recievedPacket.Read(typeof(int));;
						__level=(int) __recievedPacket.Read(typeof(int));;
						__isRegisteredUser=(bool) __recievedPacket.Read(typeof(bool));;
						__playerPreferences=(string) __recievedPacket.ReadString();

						playerToAdd=new PlayerObject(__playerID,__username, __userID, __wins, __losses, __xp, __level, __isRegisteredUser, __playerPreferences);
						_playerListInGameServer.Players.Add(playerToAdd.Username,playerToAdd);

						//AddUser(__playerID, __username, __wins, __losses, __xp, __level, __isRegisteredUser, false, __playerPreferences);
					}
				}
				WriteTextToChatArea("Recieved CurrentInGamePlayerList from Game Server.");

				//now send CurrentInGamePlayerList to clients
				PlayerObject __tempPlayer;
				NetworkPacket __packet=_buildInGamePlayerList();
				_sendMessageToClient((int) PlayerID.AllPlayers, __packet, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
//					foreach (string __username in _playerList.Players.Keys) 
//					{
//						_sendMessageToClient(playerIDToUpdate, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
//					}
				break;
		}          
		__recievedPacket.Dispose();
	}
	public void _sendInGamePlayerListToPlayer(int playerID)
	{
		NetworkPacket __packet=_buildInGamePlayerList();
		_sendMessageToClient((int) playerID, __packet, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
	}
	public NetworkPacket _buildInGamePlayerList()
	{
		NetworkPacket __returnPacket = new NetworkPacket();
		__returnPacket.Write(LobbyMessageType.CurrentInGamePlayerList);
		PlayerObject __tempPlayer;
		lock (_playerListInGameServer)
		{
			foreach (string __key in _playerListInGameServer.Players.Keys) 
			{
				__tempPlayer=(PlayerObject) _playerListInGameServer.Players[__key];
				__returnPacket.Write(__tempPlayer.PlayerID); 
				__returnPacket.Write(__tempPlayer.Username); 
				//			__returnPacket.Write(__tempPlayer.PlayerRating); 
				//			__returnPacket.Write(__tempPlayer.PlayerRank); 
				__returnPacket.Write(__tempPlayer.Wins); 
				__returnPacket.Write(__tempPlayer.Losses); 
				__returnPacket.Write(__tempPlayer.XP); 
				__returnPacket.Write(__tempPlayer.Level); 
				__returnPacket.Write(__tempPlayer.IsRegisteredUser); 
				__returnPacket.Write(__tempPlayer.PreferencesString); 
			}
		}
		return __returnPacket;
	}
	#endregion

	#region Event Handlers
	private void DataReceive(object sender, ReceiveEventArgs e)
	{
		string __caseBranchName="";

		try
		{
			NetworkPacket __recievedPacket = e.Message.ReceiveData; //get the data sent to the server 
			NetworkPacket __returnPacket = null; 

			LobbyMessageType __messageType = (LobbyMessageType)e.Message.ReceiveData.Read(typeof(LobbyMessageType));

			PlayerObject __playerObject;
			PlayerObject __playerObjectSender;

			int __playerID;
			int __playerIDSender;

			string __chatMessage;
			string __username; 
			int __recivingPlayerID;
			int __gameID;
			bool __isRanked;
			UInt32 __modChecksum=0;
			string __playerPreferences="";
			//let's lock this, so that any near-simultanious add/removes are syncronized correctly
			lock (this) 
			{
				try 
				{
					switch (__messageType) 
					{
						case LobbyMessageType.Login:
							__caseBranchName="LobbyMessageType.Login";
							__playerID=(int) e.Message.SenderID;
							__username=(string) e.Message.ReceiveData.ReadString();
							string __password=(string) e.Message.ReceiveData.ReadString();
							string __productName = e.Message.ReceiveData.ReadString();
							string __buildNumber = e.Message.ReceiveData.ReadString();
							bool __isUsingDemoClient = (bool) e.Message.ReceiveData.Read(typeof(bool));

							try 
							{
								__playerPreferences = e.Message.ReceiveData.ReadString();
							} 
							catch  
							{
								__playerPreferences = "";
							}

							//TODO: make sure the client is connecting with a correct version. 
							if (!_dbquery.IsValidGameVersion(__productName, __buildNumber))
							{
								__returnPacket = new NetworkPacket();
								__returnPacket.Write(LobbyMessageType.OutdatedVersion);
								_sendMessageToClient(__playerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);  
								__returnPacket.Dispose();                              
								return;
							}

							WriteTextToChatArea(__username + " is trying to log in.");

							//check to see if the user is logged in already
							if (_playerList.Players.ContainsKey(__username))
							{
								__returnPacket = new NetworkPacket();
								__returnPacket.Write(LobbyMessageType.LoggedInAlready);
								_sendMessageToClient(__playerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);  
								__returnPacket.Dispose();                              
								return;
							}

							// -1 = user not found
							//  0 = bad password
							// 1+ = authenticated
							// Authenticate the new lobby user, and log him in if authenticated
							int __authUserID = -1;

							LobbyMessageType __authResult = _dbquery.AuthenticateUser(
								__username, __password, true, out __authUserID);

						switch (__authUserID)
						{
							case -1:
							case 0:
								//send a login failed response
								WriteTextToChatArea(__username + " was rejected for " + __authResult.ToString() + "...");
								__returnPacket = new NetworkPacket();
								__returnPacket.Write(__authResult);
								_sendMessageToClient(__playerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
								__returnPacket.Dispose();
								break;

								// The value must be 1+
							default:
								//get the user info for this player
								__playerObject = _createPlayerObjectForPlayer(__playerID, __username, __authUserID, __playerPreferences, __isUsingDemoClient);

								//send a login approved response
								HandleSynchForLoggedInUser(__playerObject);
	                        
								//tell all logged-in users that a new user connected
								__returnPacket = new NetworkPacket();
								__returnPacket.Write(LobbyMessageType.UserLogon);
								__returnPacket.Write(__playerObject.PlayerID);
								__returnPacket.Write(__playerObject.Username);
								__returnPacket.Write(__playerObject.Wins);
								__returnPacket.Write(__playerObject.Losses);
								__returnPacket.Write(__playerObject.XP);
								__returnPacket.Write(__playerObject.Level);
								__returnPacket.Write(__playerObject.IsRegisteredUser);
								__returnPacket.Write(__playerObject.PreferencesString);
								_sendMessageToClient((int) PlayerID.AllPlayers, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
								__returnPacket.Dispose();

								_sendInGamePlayerListToPlayer(__playerID);
								break;
						}
	                    
							break;
						case LobbyMessageType.MessageToAll:
							__caseBranchName="LobbyMessageType.MessageToAll";
							__playerID=(int) e.Message.SenderID;
							__chatMessage = e.Message.ReceiveData.ReadString();
							__returnPacket = new NetworkPacket();
							__returnPacket.Write(LobbyMessageType.MessageToAll);
							__returnPacket.Write(__playerID);
							__returnPacket.Write(__chatMessage);
							WriteTextToChatArea(_playerList.GetPlayerByID(__playerID).Username + ":  " + __chatMessage);
							_sendMessageToClient((int) PlayerID.AllPlayers , __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
							__returnPacket.Dispose();
							break;
						case LobbyMessageType.MessageToUser:
							__caseBranchName="LobbyMessageType.MessageToUser";
							__playerID=(int) e.Message.SenderID;
							__username = e.Message.ReceiveData.ReadString();
							__chatMessage = e.Message.ReceiveData.ReadString();

							if (_playerList.Players.ContainsKey(__username)) 
							{
								__playerObject=(PlayerObject) _playerList.Players[__username];
								__recivingPlayerID = __playerObject.PlayerID;

								__returnPacket = new NetworkPacket();
								__returnPacket.Write(LobbyMessageType.MessageToUser);
								__returnPacket.Write(__playerID);
								__returnPacket.Write(__chatMessage);
								WriteTextToChatArea(_playerList.GetPlayerByID(__playerID).Username + ":  " + __chatMessage);
								_sendMessageToClient(__recivingPlayerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
								__returnPacket.Dispose();
							} 
							else 
							{
								//something bad happened, and we don't see that user around.  Gracefully tell the user this.
								__returnPacket = new NetworkPacket();
								__returnPacket.Write(LobbyMessageType.MessageToUserFailed);
								__returnPacket.Write(__username);
								__returnPacket.Write(__chatMessage);
								WriteTextToChatArea(_playerList.GetPlayerByID(__playerID).Username + "'s message to " + __username + " failed:  " + __chatMessage);
								_sendMessageToClient(__playerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
								__returnPacket.Dispose();
							}
							break;
						case LobbyMessageType.ChallengePlayerToGame:
							__caseBranchName="LobbyMessageType.ChallengePlayerToGame";
							__playerID=(int) e.Message.SenderID;
							__username = e.Message.ReceiveData.ReadString();
							int __gameTypeID = (int) e.Message.ReceiveData.Read(typeof(int));
							__isRanked = (bool) e.Message.ReceiveData.Read(typeof(bool));
							__modChecksum = (UInt32) e.Message.ReceiveData.Read(typeof(UInt32));

						switch(_canPlayerAcceptChallenge(__username)) 
						{
							case challengedPlayerStatus.AlreadyChallenged:
								_rejectChallenge_AlreadyChallenged(__playerID, __username);
								break;
							case challengedPlayerStatus.AlreadyChallenging:
								_rejectChallenge_AlreadyChallenging(__playerID, __username);
								break;
							case challengedPlayerStatus.PlayerNotLoggedOn:
								_rejectChallenge_TargetNotLoggedIn(__playerID, __username);
								break;
							case challengedPlayerStatus.OkayToChallenge:
								//challenge should be put through to client!  Let's look up the recipient's ID                                    
								__playerObject=(PlayerObject) _playerList.Players[__username];
								__recivingPlayerID = __playerObject.PlayerID;
								__playerObjectSender =_playerList.GetPlayerByID(__playerID);

								//let's send the packet.
								__returnPacket = new NetworkPacket();
								__returnPacket.Write(LobbyMessageType.ChallengePlayerToGame);
								__returnPacket.Write(__playerObjectSender.Username); //ChallengingPlayerUsername
								__returnPacket.Write(__gameTypeID);  //GameTypeID 
								__returnPacket.Write(__isRanked);  //isRanked
								__returnPacket.Write(__modChecksum);  //modChecksum
								_sendMessageToClient(__recivingPlayerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
								__returnPacket.Dispose();

								//let's record that this challenge is taking place.
								ChallengeInfo __tempChallengeInfo = new ChallengeInfo (__playerObjectSender, __playerObject); 
								ChallengeInfo.Challenges.Add(__playerObjectSender.Username.ToLower(),__tempChallengeInfo);        
	                    
								//let's make a note in the console:
								WriteTextToChatArea(__playerObjectSender.Username + " is challenging " + __playerObject.Username + " to a game.");

								break;
						}
							break;
						case LobbyMessageType.ChallengeCancelled:
							__caseBranchName="LobbyMessageType.ChallengeCancelled";
							_cancelAnyOutstandingChallengesForPlayer(e.Message.SenderID);
							//_notifyPlayerOfCancelledChallenge(int playerID, ChallengeInfo challenge)
							break;
						case LobbyMessageType.MismatchedModChecksum:
							__caseBranchName="LobbyMessageType.MismatchedModChecksum";
							__playerID=(int) e.Message.SenderID;
	            
							__playerObject=ChallengeInfo.GetChallengingPlayerFromChallengedPlayer(_playerList.GetPlayerByID(__playerID).Username);
							if (__playerObject!=null) 
							{
								//let's tell the unlucky challenging player AND confirm with the cancelling player.
								__returnPacket = new NetworkPacket();
								__returnPacket.Write(LobbyMessageType.MismatchedModChecksum);
								_sendMessageToClient(__playerObject.PlayerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
								_sendMessageToClient(__playerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
								__returnPacket.Dispose();

								//now let's get rid of the challenge.
								ChallengeInfo.Challenges.Remove(__playerObject.Username.ToLower());
	                        
								//let's echo to console
								WriteTextToChatArea(_playerList.GetPlayerByID(__playerID).Username + "'s mod checksum didn't match " + __playerObject.Username + "'s.  The challenge is being cancelled.");   
							} 
							else 
							{
								//This is bad -- it means they couldn't find a challenge for that player?  Let's just ignore it and hope it goes away, but let's make a note in the console:
								WriteTextToChatArea(_playerList.GetPlayerByID(__playerID).Username + "'s mod checksum didn't match " + __playerObject.Username + "'s, but we couldn't find their challenge!  THIS IS BAD");

							}
							break;						
						case LobbyMessageType.ChallengeDenied:
							__caseBranchName="LobbyMessageType.ChallengeDenied";
							__playerID=(int) e.Message.SenderID;
	            
							__playerObject=ChallengeInfo.GetChallengingPlayerFromChallengedPlayer(_playerList.GetPlayerByID(__playerID).Username);
							if (__playerObject!=null) 
							{
								//let's tell the unlucky challenging player.
								__returnPacket = new NetworkPacket();
								__returnPacket.Write(LobbyMessageType.ChallengeDenied);
								_sendMessageToClient(__playerObject.PlayerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
								__returnPacket.Dispose();

								//now let's get rid of the challenge.
								ChallengeInfo.Challenges.Remove(__playerObject.Username.ToLower());
	                        
								//let's echo to console
								WriteTextToChatArea(_playerList.GetPlayerByID(__playerID).Username + " is declining a challenge from " + __playerObject.Username + ".");   
							} 
							else 
							{
								//This is bad -- it means they couldn't find a challenge for that player?  Let's just ignore it and hope it goes away, but let's make a note in the console:
								WriteTextToChatArea(_playerList.GetPlayerByID(__playerID).Username + " is declining a challenge, but we couldn't find their challenge!  THIS IS BAD");

							}
							break;
						case LobbyMessageType.ChallengeAccepted:
							__caseBranchName="LobbyMessageType.ChallengeAccepted";
							//get challenged player info.
							__playerIDSender=(int) e.Message.SenderID;                    
							__playerObjectSender=_playerList.GetPlayerByID(__playerIDSender);

							//get the challenging player's info
							__playerObject=ChallengeInfo.GetChallengingPlayerFromChallengedPlayer(_playerList.GetPlayerByID(__playerIDSender).Username);
							__playerID=__playerObject.PlayerID;                    

							//let's sanity check that the challenging player is still around.
							if (__playerObject==null) 
							{
								//This is bad.  it seems like the challenge doesn't exist or something.  let's gracefully react to this.

								//let's cancel the challenge on the server  (I hope the other player's still there.)
								ChallengeInfo.Challenges.Remove(__playerObjectSender.Username.ToLower());

								return;                                
							}

							//let's sanity check that the challenged player is still around.
							if (__playerObjectSender==null) 
							{
								//This is bad.  the challenged player is somehow now set to nothing.  let's gracefully react to this.

								//TODO:  gracefully tell the clients what happened.

								//let's cancel the challenge on the server
								ChallengeInfo.Challenges.Remove(__playerObject.Username.ToLower());

								return;                                
							}

							//well, everything seems to be ready to being the game.  let's send them the confirmation message and the ID and get this party started.
							bool __isRated = false;	// do we have this yet?
							__gameID = _dbquery.CreateNewGame(__playerObject.UserID, __playerObjectSender.UserID, _gameTypeID, __isRated);

							//let's tell the challenging player first
							__returnPacket = new NetworkPacket();
							__returnPacket.Write(LobbyMessageType.ChallengeAccepted);
							__returnPacket.Write(__gameID);
							__returnPacket.Write(ClientRole.Challenger);
							_sendMessageToClient(__playerObject.PlayerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
							__returnPacket.Dispose();

							//now let's tell the challenged player.
							__returnPacket = new NetworkPacket();
							__returnPacket.Write(LobbyMessageType.ChallengeAccepted);
							__returnPacket.Write(__gameID);
							__returnPacket.Write(ClientRole.Challengee);
							_sendMessageToClient(__playerObjectSender.PlayerID, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
							__returnPacket.Dispose();

							string __challengeeUsername=__playerObjectSender.Username;
							int __challengeePlayerID=__playerObjectSender.PlayerID;
							string __challengerUsername=__playerObject.Username;
							int __challengerPlayerID=__playerObject.PlayerID;

							_sendPlayerOffToGame(__playerObject, false);
							_sendPlayerOffToGame(__playerObjectSender, false);

							_sendSystemMessageToAllClients(__challengeeUsername + " has accepted a challenge from " + __challengerUsername + ".");
							_tellAllPlayersAboutUserLogoff(__challengerPlayerID,__challengerUsername);
							_tellAllPlayersAboutUserLogoff(__challengeePlayerID,__challengeeUsername);

							//now let's get rid of the challenge.
							ChallengeInfo.Challenges.Remove(__playerObject.Username.ToLower());

							//let's echo to console
							WriteTextToChatArea(__playerObjectSender.Username + " has accepted a challenge from " + __playerObject.Username + ".");   

							break;
						case LobbyMessageType.PlayerPreferences:
							__caseBranchName="LobbyMessageType.PlayerPreferences";
							lock (_playerList) 
							{
								__playerID=(int) e.Message.SenderID;
								__chatMessage = e.Message.ReceiveData.ReadString();
								__returnPacket = new NetworkPacket();
								__returnPacket.Write(LobbyMessageType.PlayerPreferences);
								__returnPacket.Write(__playerID);
								__returnPacket.Write(__chatMessage);
								WriteTextToChatArea(_playerList.GetPlayerByID(__playerID).Username + ":  (Player Preferences Message) " + __chatMessage);
								_sendMessageToClient((int) PlayerID.AllPlayers, __returnPacket, PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
								__returnPacket.Dispose();
								//update the player collection
								__playerObject=(PlayerObject) _playerList.GetPlayerByID(__playerID);
								__playerObject.PreferencesString=__chatMessage;
							}
							break;
					}   
				}            
				catch (Exception ex)
				{
					_logException(ex, "LobbyServer DataReceive(), branch=" +__caseBranchName); 
				}
			}                            
			__recievedPacket.Dispose();
		}
		catch (Exception ex)
		{
			SI_MiddleTier.MiddleTier.RecordExceptionToErrorFile("Lobby Server", ex, "LobbyServer DataReceive(), branch=" +__caseBranchName);  //make sure the name of the server is listed -- it will be recorded in the log file.

			throw ex;
		}
		//check to make sure we're either connecting to game server (or trying), and if not, let's connect. 
		//now we need to connect to the Game Server to get player lists.
		if (_connectedToGameServer==false && _tryingToConnectToGameServer==false)
		{
			Thread GameServerConnectThread = new Thread(new ThreadStart(_connectToGameServerAndShakeHands));
			//GameServerConnectThread.Priority = System.Threading.ThreadPriority.Normal;
			GameServerConnectThread.Start();
			

		}

	}
	private void PlayerLeft(object sender, PlayerDestroyedEventArgs e)
	{
		try
		{
			//A player left -- let's handle that
			if (_playerList.PlayerIDExists(e.Message.PlayerID)) 
			{                
				_cancelAnyOutstandingChallengesForPlayer(e.Message.PlayerID);

				string __username = _playerList.GetPlayerByID(e.Message.PlayerID).Username;
				WriteTextToChatArea(__username + " has logged off.");

				RemoveUser(e.Message.PlayerID);

				if (_playerList.Players.Count>0) 
				{
					_tellAllPlayersAboutUserLogoff(e.Message.PlayerID,__username);
				}
			}
		}
		catch (Exception ex)
		{
			SI_MiddleTier.MiddleTier.RecordExceptionToErrorFile("Lobby Server", ex, "Problem in PlayerLeft()");  //make sure the name of the server is listed -- it will be recorded in the log file.

			throw ex;
		}
	}

	public virtual void ShutDown()
	{
		//            foreach (PlayerObject __playerObject in _playerList.Players.Values) 
		//            {
		//            }
		_lobbyServer.CancelAsyncOperation(CancelFlags.AllOperations);
		if (!_lobbyServer.Disposed)
		{
			_lobbyServer.Dispose();
		}
	}
	#endregion
}

